<?php
/**
 * Admin-Einstellungsseite fuer Grid Connector
 *
 * @package WP_Grid_Connector
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class WPGrid_Settings_Page {

    private function get_lang(): string {
        $saved = get_option( 'wpgrid_lang', '' );
        if ( $saved ) return $saved;
        $wp_locale = get_locale();
        return ( strpos( $wp_locale, 'de' ) === 0 ) ? 'de' : 'en';
    }

    private function t( string $key ): string {
        $lang = $this->get_lang();
        $strings = array(
            'de' => array(
                'api_connection'    => 'API-Verbindung',
                'status'            => 'Status',
                'connected'         => '&#9679; Verbunden',
                'not_connected'     => '&#9679; Nicht verbunden',
                'api_key'           => 'API-Key',
                'save'              => 'Speichern',
                'saving'            => 'Speichere...',
                'saved'             => 'Gespeichert!',
                'api_key_desc'      => 'Kopiere den API-Key aus dem WP-Grid Dashboard (Projekt → WordPress-Tab) und füge ihn hier ein.',
                'endpoint_url'      => 'Endpoint URL',
                'setup'             => 'Einrichtung',
                'setup_1'           => 'Öffne dein WP-Grid Dashboard unter wp-grid.de',
                'setup_2'           => 'Gehe zu deinem Projekt → WordPress-Tab',
                'setup_3'           => 'Kopiere den angezeigten API-Key',
                'setup_4'           => 'Füge den Key oben ein und klicke "Speichern"',
                'setup_5'           => 'Klicke im WP-Grid Dashboard auf "Verbindung prüfen"',
                'plugin_version'    => 'Plugin-Version',
                'installed_version' => 'Installierte Version',
                'update_hint'       => 'Updates werden über die WordPress Plugin-Verwaltung eingespielt.',
                'to_plugins'        => 'Zur Plugin-Verwaltung →',
                'invalid_key'       => 'Bitte einen gültigen API-Key eingeben (beginnt mit wpgk_).',
                'error'             => 'Fehler',
                'unknown'           => 'Unbekannt',
                'no_permission'     => 'Keine Berechtigung.',
                'invalid_key_msg'   => 'Ungültiger API-Key. Der Key muss mit wpgk_ beginnen.',
                'key_saved'         => 'API-Key erfolgreich gespeichert.',
                'try_now'           => 'Jetzt kostenlos testen →',
                'hero_title'        => 'WordPress Management',
                'hero_highlight'    => 'neu gedacht.',
                'hero_desc'         => 'Verwalte alle deine WordPress-Websites zentral – Plugins, Backups, Performance, FTP und mehr. Alles in einem Dashboard.',
                'lang_label'        => 'Sprache',
            ),
            'en' => array(
                'api_connection'    => 'API Connection',
                'status'            => 'Status',
                'connected'         => '&#9679; Connected',
                'not_connected'     => '&#9679; Not connected',
                'api_key'           => 'API Key',
                'save'              => 'Save',
                'saving'            => 'Saving...',
                'saved'             => 'Saved!',
                'api_key_desc'      => 'Copy the API key from your WP-Grid Dashboard (Project → WordPress tab) and paste it here.',
                'endpoint_url'      => 'Endpoint URL',
                'setup'             => 'Setup',
                'setup_1'           => 'Open your WP-Grid Dashboard at wp-grid.de',
                'setup_2'           => 'Go to your project → WordPress tab',
                'setup_3'           => 'Copy the displayed API key',
                'setup_4'           => 'Paste the key above and click "Save"',
                'setup_5'           => 'Click "Check connection" in your WP-Grid Dashboard',
                'plugin_version'    => 'Plugin Version',
                'installed_version' => 'Installed Version',
                'update_hint'       => 'Updates are installed via the WordPress plugin manager.',
                'to_plugins'        => 'Go to Plugin Manager →',
                'invalid_key'       => 'Please enter a valid API key (starts with wpgk_).',
                'error'             => 'Error',
                'unknown'           => 'Unknown',
                'no_permission'     => 'No permission.',
                'invalid_key_msg'   => 'Invalid API key. The key must start with wpgk_.',
                'key_saved'         => 'API key saved successfully.',
                'try_now'           => 'Try for free →',
                'hero_title'        => 'WordPress Management',
                'hero_highlight'    => 'reimagined.',
                'hero_desc'         => 'Manage all your WordPress websites centrally – plugins, backups, performance, FTP and more. All in one dashboard.',
                'lang_label'        => 'Language',
            ),
        );
        return $strings[ $lang ][ $key ] ?? $key;
    }

    public function menue_hinzufuegen() {
        add_options_page(
            __( 'Grid Connector', 'grid-connector' ),
            __( 'Grid Connector', 'grid-connector' ),
            'manage_options',
            'grid-connector',
            array( $this, 'seite_rendern' )
        );
        add_action( 'admin_head', array( $this, 'menu_icon_css' ) );
    }

    public function menu_icon_css() {
        $icon_url = WPGRID_PLUGIN_URL . 'assets/nurlogo.svg';
        ?>
        <style>
        #adminmenu .toplevel_page_wp-grid-connector .wp-menu-image img,
        #adminmenu li a[href*="grid-connector"] .wp-menu-image img {
            width: 20px !important;
            height: 20px !important;
            padding: 0 !important;
        }
        #adminmenu li a[href*="grid-connector"] .wp-menu-image::before {
            content: none !important;
        }
        #adminmenu li a[href*="grid-connector"] .wp-menu-image {
            background-image: url('<?php echo esc_url( $icon_url ); ?>') !important;
            background-repeat: no-repeat !important;
            background-position: center !important;
            background-size: 18px 18px !important;
        }
        </style>
        <?php
    }

    public function scripts_laden( $hook ) {
        if ( 'settings_page_grid-connector' !== $hook ) {
            return;
        }
        wp_enqueue_style(
            'wpgrid-admin',
            WPGRID_PLUGIN_URL . 'admin/admin.css',
            array(),
            WPGRID_VERSION
        );
    }

    // Fremde Admin-Notices auf unserer Seite blockieren
    public function notices_blockieren() {
        $screen = get_current_screen();
        if ( $screen && 'settings_page_grid-connector' === $screen->id ) {
            remove_all_actions( 'admin_notices' );
            remove_all_actions( 'all_admin_notices' );
            // Eigene Notices wieder zulassen (PUC)
            add_action( 'all_admin_notices', array( $this, 'eigene_notices' ) );
        }
    }

    public function eigene_notices() {
        // Platzhalter – nur unsere eigenen Notices hier
    }

    // AJAX: Sprache speichern
    public function ajax_lang_speichern() {
        check_ajax_referer( 'wpgrid_save_lang', 'nonce' );
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error();
        }
        $lang = isset( $_POST['lang'] ) ? sanitize_key( wp_unslash( $_POST['lang'] ) ) : 'de';
        if ( ! in_array( $lang, array( 'de', 'en' ), true ) ) {
            $lang = 'de';
        }
        update_option( 'wpgrid_lang', $lang );
        wp_send_json_success( array( 'lang' => $lang ) );
    }

    /**
     * AJAX: API-Key speichern
     */
    public function ajax_key_regenerieren() {
        check_ajax_referer( 'wpgrid_regenerate_key', 'nonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( array( 'message' => $this->t( 'no_permission' ) ) );
        }

        $new_key = isset( $_POST['api_key'] ) ? sanitize_text_field( wp_unslash( $_POST['api_key'] ) ) : '';
        if ( empty( $new_key ) || strpos( $new_key, 'wpgk_' ) !== 0 ) {
            wp_send_json_error( array( 'message' => $this->t( 'invalid_key_msg' ) ) );
        }

        update_option( WPGRID_OPTION_KEY, $new_key, false );

        wp_send_json_success( array(
            'key'     => $new_key,
            'message' => $this->t( 'key_saved' ),
        ) );
    }

    public function seite_rendern() {
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_die( esc_html( $this->t( 'no_permission' ) ) );
        }

        $api_key   = get_option( WPGRID_OPTION_KEY, '' );
        $has_key   = ! empty( $api_key ) && strpos( $api_key, 'wpgk_' ) === 0;
        $nonce     = wp_create_nonce( 'wpgrid_regenerate_key' );
        $lang_nonce = wp_create_nonce( 'wpgrid_save_lang' );
        $ajax_url  = admin_url( 'admin-ajax.php' );
        $logo_url  = WPGRID_PLUGIN_URL . 'assets/nurlogo.svg';
        $bg_url    = WPGRID_PLUGIN_URL . 'assets/dashboard-verpixel.jpg';
        $lang      = $this->get_lang();
        ?>
        <div class="wrap wpgrid-wrap">

            <!-- Hero Banner -->
            <div class="wpgrid-hero" style="background-image: url('<?php echo esc_url( $bg_url ); ?>');">
                <div class="wpgrid-hero-overlay"></div>
                <div class="wpgrid-hero-content">
                    <img src="<?php echo esc_url( $logo_url ); ?>" alt="WP-Grid" class="wpgrid-hero-logo" />
                    <div class="wpgrid-hero-text">
                        <h1 class="wpgrid-hero-title"><?php echo esc_html( $this->t( 'hero_title' ) ); ?><br><span class="wpgrid-hero-highlight"><?php echo esc_html( $this->t( 'hero_highlight' ) ); ?></span></h1>
                        <p class="wpgrid-hero-tagline">Stop switching tabs.</p>
                        <p class="wpgrid-hero-desc"><?php echo esc_html( $this->t( 'hero_desc' ) ); ?></p>
                        <a href="https://wp-grid.de" target="_blank" rel="noopener" class="wpgrid-hero-btn">
                            <?php echo esc_html( $this->t( 'try_now' ) ); ?>
                        </a>
                    </div>
                    <div class="wpgrid-hero-meta">
                        <span class="wpgrid-version-badge">v<?php echo esc_html( WPGRID_VERSION ); ?></span>
                        <!-- Sprachumschalter -->
                        <div class="wpgrid-lang-switcher">
                            <button
                                class="wpgrid-lang-btn <?php echo $lang === 'de' ? 'active' : ''; ?>"
                                data-lang="de"
                                data-nonce="<?php echo esc_attr( $lang_nonce ); ?>"
                                data-ajax="<?php echo esc_url( $ajax_url ); ?>"
                                title="Deutsch"
                            >DE</button>
                            <button
                                class="wpgrid-lang-btn <?php echo $lang === 'en' ? 'active' : ''; ?>"
                                data-lang="en"
                                data-nonce="<?php echo esc_attr( $lang_nonce ); ?>"
                                data-ajax="<?php echo esc_url( $ajax_url ); ?>"
                                title="English"
                            >EN</button>
                        </div>
                    </div>
                </div>
            </div>

            <!-- API-Key Sektion -->
            <div class="wpgrid-card">
                <h2><?php echo esc_html( $this->t( 'api_connection' ) ); ?></h2>
                <table class="form-table wpgrid-table">
                    <tr>
                        <th><?php echo esc_html( $this->t( 'status' ) ); ?></th>
                        <td>
                            <?php if ( $has_key ) : ?>
                                <span class="wpgrid-status wpgrid-status--aktiv">
                                    <?php echo wp_kses( $this->t( 'connected' ), array() ); ?>
                                </span>
                            <?php else : ?>
                                <span class="wpgrid-status wpgrid-status--inaktiv">
                                    <?php echo wp_kses( $this->t( 'not_connected' ), array() ); ?>
                                </span>
                            <?php endif; ?>
                        </td>
                    </tr>
                    <tr>
                        <th><label for="wpgrid-api-key"><?php echo esc_html( $this->t( 'api_key' ) ); ?></label></th>
                        <td>
                            <div class="wpgrid-key-row">
                                <input
                                    type="text"
                                    id="wpgrid-api-key"
                                    value="<?php echo esc_attr( $api_key ); ?>"
                                    class="regular-text wpgrid-key-input"
                                    placeholder="wpgk_..."
                                    aria-label="<?php echo esc_attr( $this->t( 'api_key' ) ); ?>"
                                />
                                <button
                                    type="button"
                                    id="wpgrid-save-key"
                                    class="button button-primary"
                                    data-nonce="<?php echo esc_attr( $nonce ); ?>"
                                    data-ajax="<?php echo esc_url( $ajax_url ); ?>"
                                    data-txt-save="<?php echo esc_attr( $this->t( 'save' ) ); ?>"
                                    data-txt-saving="<?php echo esc_attr( $this->t( 'saving' ) ); ?>"
                                    data-txt-saved="<?php echo esc_attr( $this->t( 'saved' ) ); ?>"
                                    data-txt-invalid="<?php echo esc_attr( $this->t( 'invalid_key' ) ); ?>"
                                    data-txt-error="<?php echo esc_attr( $this->t( 'error' ) ); ?>"
                                    data-txt-unknown="<?php echo esc_attr( $this->t( 'unknown' ) ); ?>"
                                >
                                    <?php echo esc_html( $this->t( 'save' ) ); ?>
                                </button>
                            </div>
                            <p class="description"><?php echo esc_html( $this->t( 'api_key_desc' ) ); ?></p>
                        </td>
                    </tr>
                    <tr>
                        <th><?php echo esc_html( $this->t( 'endpoint_url' ) ); ?></th>
                        <td><code class="wpgrid-endpoint"><?php echo esc_html( get_rest_url( null, WPGRID_API_NS ) ); ?></code></td>
                    </tr>
                </table>
            </div>

            <!-- Anleitung -->
            <div class="wpgrid-card">
                <h2><?php echo esc_html( $this->t( 'setup' ) ); ?></h2>
                <ol style="padding-left: 20px; line-height: 2;">
                    <li><?php echo esc_html( $this->t( 'setup_1' ) ); ?></li>
                    <li><?php echo esc_html( $this->t( 'setup_2' ) ); ?></li>
                    <li><?php echo esc_html( $this->t( 'setup_3' ) ); ?></li>
                    <li><?php echo esc_html( $this->t( 'setup_4' ) ); ?></li>
                    <li><?php echo esc_html( $this->t( 'setup_5' ) ); ?></li>
                </ol>
            </div>

            <!-- Version Info -->
            <div class="wpgrid-card">
                <h2><?php echo esc_html( $this->t( 'plugin_version' ) ); ?></h2>
                <table class="form-table wpgrid-table">
                    <tr>
                        <th><?php echo esc_html( $this->t( 'installed_version' ) ); ?></th>
                        <td><code><?php echo esc_html( WPGRID_VERSION ); ?></code></td>
                    </tr>
                </table>
                <p class="description" style="margin-top:8px;">
                    <?php echo esc_html( $this->t( 'update_hint' ) ); ?>
                    <a href="<?php echo esc_url( admin_url( 'plugins.php' ) ); ?>"><?php echo esc_html( $this->t( 'to_plugins' ) ); ?></a>
                </p>
            </div>

        </div>

        <script>
        (function() {
            // API-Key speichern
            var saveBtn = document.getElementById("wpgrid-save-key");
            if (saveBtn) {
                saveBtn.addEventListener("click", function() {
                    var keyInput = document.getElementById("wpgrid-api-key");
                    var key = keyInput.value.trim();
                    if (!key || key.indexOf("wpgk_") !== 0) {
                        alert(saveBtn.dataset.txtInvalid || "Invalid API key");
                        return;
                    }
                    saveBtn.disabled = true;
                    saveBtn.textContent = saveBtn.dataset.txtSaving || "Saving...";
                    var xhr = new XMLHttpRequest();
                    xhr.open("POST", saveBtn.dataset.ajax);
                    xhr.setRequestHeader("Content-Type", "application/x-www-form-urlencoded");
                    xhr.onload = function() {
                        try {
                            var clean = xhr.responseText.replace(/^\uFEFF+/, "");
                            var resp = JSON.parse(clean);
                            if (resp.success) {
                                saveBtn.textContent = saveBtn.dataset.txtSaved || "Saved!";
                                setTimeout(function() {
                                    saveBtn.textContent = saveBtn.dataset.txtSave || "Save";
                                    location.reload();
                                }, 1500);
                            } else {
                                alert((saveBtn.dataset.txtError || "Error") + ": " + (resp.data ? resp.data.message : (saveBtn.dataset.txtUnknown || "Unknown")));
                                saveBtn.textContent = saveBtn.dataset.txtSave || "Save";
                            }
                        } catch(e) {
                            alert("Parse error: " + e.message);
                            saveBtn.textContent = saveBtn.dataset.txtSave || "Save";
                        }
                        saveBtn.disabled = false;
                    };
                    xhr.send("action=wpgrid_regenerate_key&nonce=" + encodeURIComponent(saveBtn.dataset.nonce) + "&api_key=" + encodeURIComponent(key));
                });
            }

            // Sprachumschalter
            document.querySelectorAll(".wpgrid-lang-btn").forEach(function(btn) {
                btn.addEventListener("click", function() {
                    var lang = btn.dataset.lang;
                    var xhr = new XMLHttpRequest();
                    xhr.open("POST", btn.dataset.ajax);
                    xhr.setRequestHeader("Content-Type", "application/x-www-form-urlencoded");
                    xhr.onload = function() {
                        location.reload();
                    };
                    xhr.send("action=wpgrid_save_lang&nonce=" + encodeURIComponent(btn.dataset.nonce) + "&lang=" + encodeURIComponent(lang));
                });
            });
        })();
        </script>
        <?php
    }

    private function letzte_logs_laden( $anzahl = 10 ) {
        global $wpdb;
        $tabelle = $wpdb->prefix . WPGRID_LOG_TABLE;

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name is built from $wpdb->prefix, safe
        $tabellen_existiert = $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $tabelle ) );
        if ( ! $tabellen_existiert ) {
            return array();
        }

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
        return $wpdb->get_results(
            $wpdb->prepare(
                // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                "SELECT * FROM {$tabelle} ORDER BY zeitpunkt DESC LIMIT %d",
                absint( $anzahl )
            )
        );
    }
}