<?php
/**
 * Plugin Name:       Grid Connector
 * Plugin URI:        https://wp-grid.de
 * Description:       Connects your WordPress website to the WP-Grid Dashboard for centralized management, monitoring, and updates.
 * Version:           1.2.4
 * Requires at least: 6.0
 * Requires PHP:      7.4
 * Author:            Meindl Webdesign
 * Author URI:        https://meindl-webdesign.de
 * License:           GPL v2 or later
 * License URI:       https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain:       grid-connector
 * Network:           true
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

// BOM-Cleanup: Entfernt unsichtbare BOM-Zeichen aus AJAX-Responses
if ( defined( 'DOING_AJAX' ) && DOING_AJAX ) {
    ob_start( function( $output ) {
        return ltrim( $output, "\xEF\xBB\xBF" );
    } );
}

define( 'WPGRID_VERSION',     '1.2.4' );
define( 'WPGRID_PLUGIN_DIR',  plugin_dir_path( __FILE__ ) );
define( 'WPGRID_PLUGIN_URL',  plugin_dir_url( __FILE__ ) );
define( 'WPGRID_PLUGIN_FILE', __FILE__ );

define( 'WPGRID_API_NS',      'wpgrid/v1' );
define( 'WPGRID_LOG_TABLE',   'wpgrid_logs' );
define( 'WPGRID_OPTION_KEY',  'wpgrid_api_key' );
define( 'WPGRID_RATE_LIMIT',  100 );

/**
 * Haupt-Plugin-Klasse (Singleton)
 */
final class WPGrid_Connector {

    private static $instance = null;

    public $auth;
    public $api;
    public $health;
    public $updates;
    public $files;
    public $snippets;
    public $backups;

    private function __construct() {
        $this->lade_abhaengigkeiten();
        $this->initialisiere_klassen();
        $this->registriere_hooks();
    }

    public static function instanz() {
        if ( null === self::$instance ) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function lade_abhaengigkeiten() {
        require_once WPGRID_PLUGIN_DIR . 'includes/class-auth.php';
        require_once WPGRID_PLUGIN_DIR . 'includes/class-api-handler.php';
        require_once WPGRID_PLUGIN_DIR . 'includes/class-health.php';
        require_once WPGRID_PLUGIN_DIR . 'includes/class-updates.php';
        require_once WPGRID_PLUGIN_DIR . 'includes/class-files.php';
        require_once WPGRID_PLUGIN_DIR . 'includes/class-snippets.php';
        require_once WPGRID_PLUGIN_DIR . 'includes/class-backups.php';

        if ( is_admin() ) {
            require_once WPGRID_PLUGIN_DIR . 'admin/settings-page.php';
        }
    }

    private function initialisiere_klassen() {
        $this->auth         = new WPGrid_Auth();
        $this->health       = new WPGrid_Health();
        $this->updates      = new WPGrid_Updates();
        $this->files        = new WPGrid_Files();
        $this->snippets     = new WPGrid_Snippets();
        $this->backups      = new WPGrid_Backups();
        $this->api          = new WPGrid_API_Handler(
            $this->auth,
            $this->health,
            $this->updates,
            $this->files,
            $this->snippets,
            $this->backups
        );
    }

    private function registriere_hooks() {
        add_action( 'rest_api_init', array( $this->api, 'registriere_routen' ) );

        if ( is_admin() ) {
            $settings = new WPGrid_Settings_Page();
            add_action( 'admin_menu',            array( $settings, 'menue_hinzufuegen' ) );
            add_action( 'admin_enqueue_scripts', array( $settings, 'scripts_laden' ) );
            add_action( 'admin_notices',         array( $settings, 'notices_blockieren' ), 1 );
            add_action( 'all_admin_notices',     array( $settings, 'notices_blockieren' ), 1 );
            add_action( 'wp_ajax_wpgrid_regenerate_key', array( $settings, 'ajax_key_regenerieren' ) );
            add_action( 'wp_ajax_wpgrid_save_lang',      array( $settings, 'ajax_lang_speichern' ) );
        }
    }
}

register_activation_hook( __FILE__, 'wpgrid_aktivierung' );

function wpgrid_aktivierung() {
    // Key wird nicht auto-generiert - User kopiert ihn aus dem WP-Grid Dashboard
    wpgrid_log_tabelle_erstellen();
    update_option( 'wpgrid_version', WPGRID_VERSION );
    flush_rewrite_rules();
}

register_deactivation_hook( __FILE__, 'wpgrid_deaktivierung' );

function wpgrid_deaktivierung() {
    flush_rewrite_rules();
}

function wpgrid_api_key_generieren() {
    return 'wpgk_' . bin2hex( random_bytes( 32 ) );
}

function wpgrid_log_tabelle_erstellen() {
    global $wpdb;
    $tabelle     = $wpdb->prefix . WPGRID_LOG_TABLE;
    $zeichensatz = $wpdb->get_charset_collate();
    $sql = "CREATE TABLE IF NOT EXISTS {$tabelle} (
        id          BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
        zeitpunkt   DATETIME            NOT NULL DEFAULT CURRENT_TIMESTAMP,
        methode     VARCHAR(10)         NOT NULL,
        endpunkt    VARCHAR(255)        NOT NULL,
        ip_adresse  VARCHAR(45)         NOT NULL,
        status_code SMALLINT(3)         NOT NULL DEFAULT 200,
        antwortzeit FLOAT               NOT NULL DEFAULT 0,
        fehler      TEXT                DEFAULT NULL,
        PRIMARY KEY (id),
        KEY idx_zeitpunkt (zeitpunkt),
        KEY idx_endpunkt  (endpunkt)
    ) {$zeichensatz};";
    require_once ABSPATH . 'wp-admin/includes/upgrade.php';
    dbDelta( $sql );
}

function wpgrid_antwort( $erfolg, $daten = null, $fehler = '' ) {
    $antwort = array( 'success' => (bool) $erfolg );
    if ( $erfolg && null !== $daten ) {
        $antwort['data'] = $daten;
    }
    if ( ! $erfolg && $fehler ) {
        $antwort['error'] = $fehler;
    }
    $status = $erfolg ? 200 : 400;
    return new WP_REST_Response( $antwort, $status );
}

function wpgrid_log_schreiben( $methode, $endpunkt, $ip, $status, $antwortzeit, $fehler = '' ) {
    global $wpdb;
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Log insert, no caching needed
    $wpdb->insert(
        $wpdb->prefix . WPGRID_LOG_TABLE,
        array(
            'methode'     => sanitize_text_field( $methode ),
            'endpunkt'    => sanitize_text_field( $endpunkt ),
            'ip_adresse'  => sanitize_text_field( $ip ),
            'status_code' => absint( $status ),
            'antwortzeit' => floatval( $antwortzeit ),
            'fehler'      => sanitize_textarea_field( $fehler ),
        ),
        array( '%s', '%s', '%s', '%d', '%f', '%s' )
    );
}

function wpgrid_debug( $nachricht, $kontext = null ) {
    if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
        $ausgabe = '[WP-Grid] ' . $nachricht;
        if ( null !== $kontext ) {
            $ausgabe .= ' | ' . wp_json_encode( $kontext );
        }
        // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
        error_log( $ausgabe );
    }
}

add_action( 'plugins_loaded', array( 'WPGrid_Connector', 'instanz' ) );


