<?php
/**
 * REST API Handler - registriert alle Routen und Endpoints
 *
 * @package WP_Grid_Connector
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class WPGrid_API_Handler {

    private $auth;
    private $health;
    private $updates;
    private $files;
    private $snippets;
    private $backups;

    public function __construct( $auth, $health, $updates, $files, $snippets, $backups = null ) {
        $this->auth     = $auth;
        $this->health   = $health;
        $this->updates  = $updates;
        $this->files    = $files;
        $this->snippets = $snippets;
        $this->backups  = $backups;
    }

    /**
     * Alle REST-Routen registrieren
     */
    public function registriere_routen() {
        $auth_cb = array( $this->auth, 'api_key_pruefen' );

        // Discovery: App braucht nur den API-Key, URL wird automatisch ermittelt.
        // Dieser Endpoint gibt die vollstaendige Connector-URL zurueck.
        register_rest_route( WPGRID_API_NS, '/discover', array(
            'methods'             => WP_REST_Server::READABLE,
            'callback'            => array( $this, 'discover' ),
            'permission_callback' => $auth_cb,
        ) );

        // Site-Informationen
        register_rest_route( WPGRID_API_NS, '/info', array(
            'methods'             => WP_REST_Server::READABLE,
            'callback'            => array( $this, 'site_info' ),
            'permission_callback' => $auth_cb,
        ) );

        // Health Checks
        register_rest_route( WPGRID_API_NS, '/health', array(
            'methods'             => WP_REST_Server::READABLE,
            'callback'            => array( $this->health, 'health_check' ),
            'permission_callback' => $auth_cb,
        ) );

        // Plugin-Verwaltung
        register_rest_route( WPGRID_API_NS, '/plugins', array(
            'methods'             => WP_REST_Server::READABLE,
            'callback'            => array( $this, 'plugins_liste' ),
            'permission_callback' => $auth_cb,
        ) );

        register_rest_route( WPGRID_API_NS, '/plugins/activate', array(
            'methods'             => WP_REST_Server::CREATABLE,
            'callback'            => array( $this, 'plugin_aktivieren' ),
            'permission_callback' => $auth_cb,
            'args'                => array(
                'slug' => array(
                    'required'          => true,
                    'sanitize_callback' => 'sanitize_text_field',
                    'validate_callback' => array( $this, 'slug_validieren' ),
                ),
            ),
        ) );

        register_rest_route( WPGRID_API_NS, '/plugins/deactivate', array(
            'methods'             => WP_REST_Server::CREATABLE,
            'callback'            => array( $this, 'plugin_deaktivieren' ),
            'permission_callback' => $auth_cb,
            'args'                => array(
                'slug' => array(
                    'required'          => true,
                    'sanitize_callback' => 'sanitize_text_field',
                    'validate_callback' => array( $this, 'slug_validieren' ),
                ),
            ),
        ) );

        register_rest_route( WPGRID_API_NS, '/plugins/install', array(
            'methods'             => WP_REST_Server::CREATABLE,
            'callback'            => array( $this, 'plugin_installieren' ),
            'permission_callback' => $auth_cb,
            'args'                => array(
                'url' => array(
                    'required'          => true,
                    'sanitize_callback' => 'esc_url_raw',
                    'validate_callback' => array( $this, 'url_validieren' ),
                ),
            ),
        ) );

        // Updates
        register_rest_route( WPGRID_API_NS, '/update/core', array(
            'methods'             => WP_REST_Server::CREATABLE,
            'callback'            => array( $this->updates, 'core_updaten' ),
            'permission_callback' => $auth_cb,
        ) );

        register_rest_route( WPGRID_API_NS, '/update/plugin', array(
            'methods'             => WP_REST_Server::CREATABLE,
            'callback'            => array( $this->updates, 'plugin_updaten' ),
            'permission_callback' => $auth_cb,
            'args'                => array(
                'slug' => array(
                    'required'          => true,
                    'sanitize_callback' => 'sanitize_text_field',
                    'validate_callback' => array( $this, 'slug_validieren' ),
                ),
            ),
        ) );

        register_rest_route( WPGRID_API_NS, '/update/theme', array(
            'methods'             => WP_REST_Server::CREATABLE,
            'callback'            => array( $this->updates, 'theme_updaten' ),
            'permission_callback' => $auth_cb,
            'args'                => array(
                'slug' => array(
                    'required'          => true,
                    'sanitize_callback' => 'sanitize_text_field',
                ),
            ),
        ) );

        register_rest_route( WPGRID_API_NS, '/update/all', array(
            'methods'             => WP_REST_Server::CREATABLE,
            'callback'            => array( $this->updates, 'alles_updaten' ),
            'permission_callback' => $auth_cb,
        ) );

        // File Manager
        register_rest_route( WPGRID_API_NS, '/files/list', array(
            'methods'             => WP_REST_Server::READABLE,
            'callback'            => array( $this->files, 'dateien_listen' ),
            'permission_callback' => $auth_cb,
            'args'                => array(
                'path' => array(
                    'default'           => '/wp-content/uploads',
                    'sanitize_callback' => 'sanitize_text_field',
                ),
            ),
        ) );

        register_rest_route( WPGRID_API_NS, '/files/upload', array(
            'methods'             => WP_REST_Server::CREATABLE,
            'callback'            => array( $this->files, 'datei_hochladen' ),
            'permission_callback' => $auth_cb,
        ) );

        register_rest_route( WPGRID_API_NS, '/files/delete', array(
            'methods'             => WP_REST_Server::DELETABLE,
            'callback'            => array( $this->files, 'datei_loeschen' ),
            'permission_callback' => $auth_cb,
            'args'                => array(
                'path' => array(
                    'required'          => true,
                    'sanitize_callback' => 'sanitize_text_field',
                ),
            ),
        ) );

        register_rest_route( WPGRID_API_NS, '/files/download', array(
            'methods'             => WP_REST_Server::READABLE,
            'callback'            => array( $this->files, 'datei_herunterladen' ),
            'permission_callback' => $auth_cb,
            'args'                => array(
                'path' => array(
                    'required'          => true,
                    'sanitize_callback' => 'sanitize_text_field',
                ),
            ),
        ) );

        // Code Snippets
        register_rest_route( WPGRID_API_NS, '/snippets', array(
            'methods'             => WP_REST_Server::READABLE,
            'callback'            => array( $this->snippets, 'snippets_listen' ),
            'permission_callback' => $auth_cb,
        ) );

        register_rest_route( WPGRID_API_NS, '/snippets/add', array(
            'methods'             => WP_REST_Server::CREATABLE,
            'callback'            => array( $this->snippets, 'snippet_hinzufuegen' ),
            'permission_callback' => $auth_cb,
            'args'                => array(
                'title' => array(
                    'required'          => true,
                    'sanitize_callback' => 'sanitize_text_field',
                ),
                'code' => array(
                    'required' => true,
                ),
            ),
        ) );

        register_rest_route( WPGRID_API_NS, '/snippets/remove', array(
            'methods'             => WP_REST_Server::DELETABLE,
            'callback'            => array( $this->snippets, 'snippet_entfernen' ),
            'permission_callback' => $auth_cb,
            'args'                => array(
                'id' => array(
                    'required'          => true,
                    'sanitize_callback' => 'absint',
                ),
            ),
        ) );

        // Report-Daten (erweitert für Kundenreports)
        register_rest_route( WPGRID_API_NS, '/report-data', array(
            'methods'             => WP_REST_Server::READABLE,
            'callback'            => array( $this, 'report_data' ),
            'permission_callback' => $auth_cb,
        ) );

        // PHP-Fehler & Error-Log Analyse
        register_rest_route( WPGRID_API_NS, '/errors', array(
            'methods'             => WP_REST_Server::READABLE,
            'callback'            => array( $this, 'fehler_analyse' ),
            'permission_callback' => $auth_cb,
        ) );

        // WP-Config Einstellungen setzen
        register_rest_route( WPGRID_API_NS, '/config/set', array(
            'methods'             => WP_REST_Server::CREATABLE,
            'callback'            => array( $this, 'config_setzen' ),
            'permission_callback' => $auth_cb,
        ) );

        // .htaccess lesen
        register_rest_route( WPGRID_API_NS, '/htaccess', array(
            'methods'             => WP_REST_Server::READABLE,
            'callback'            => array( $this, 'htaccess_lesen' ),
            'permission_callback' => $auth_cb,
        ) );

        // .htaccess schreiben
        register_rest_route( WPGRID_API_NS, '/htaccess', array(
            'methods'             => WP_REST_Server::CREATABLE,
            'callback'            => array( $this, 'htaccess_schreiben' ),
            'permission_callback' => $auth_cb,
        ) );

        // WP Core Info
        register_rest_route( WPGRID_API_NS, '/core-info', array(
            'methods'             => WP_REST_Server::READABLE,
            'callback'            => array( $this, 'core_info' ),
            'permission_callback' => $auth_cb,
        ) );

        // Secure Auto-Login: generiert einen einmaligen Login-Token
        register_rest_route( WPGRID_API_NS, '/auto-login', array(
            'methods'             => WP_REST_Server::CREATABLE,
            'callback'            => array( $this, 'auto_login_token_erstellen' ),
            'permission_callback' => $auth_cb,
        ) );

        register_rest_route( WPGRID_API_NS, '/auto-login/(?P<token>[a-f0-9]+)', array(
            'methods'             => WP_REST_Server::READABLE,
            'callback'            => array( $this, 'auto_login_ausfuehren' ),
            'permission_callback' => '__return_true',
        ) );

        // Backups
        if ( $this->backups ) {
            register_rest_route( WPGRID_API_NS, '/backups', array(
                'methods'             => WP_REST_Server::READABLE,
                'callback'            => array( $this->backups, 'backups_listen' ),
                'permission_callback' => $auth_cb,
            ) );

            register_rest_route( WPGRID_API_NS, '/backups', array(
                'methods'             => WP_REST_Server::CREATABLE,
                'callback'            => array( $this->backups, 'backup_erstellen' ),
                'permission_callback' => $auth_cb,
            ) );

            register_rest_route( WPGRID_API_NS, '/backups/(?P<backup_id>[^/]+)', array(
                'methods'             => WP_REST_Server::DELETABLE,
                'callback'            => array( $this->backups, 'backup_loeschen' ),
                'permission_callback' => $auth_cb,
                'args'                => array(
                    'backup_id' => array(
                        'required'          => true,
                        'sanitize_callback' => 'sanitize_file_name',
                    ),
                ),
            ) );

            register_rest_route( WPGRID_API_NS, '/backups/(?P<backup_id>[^/]+)/restore', array(
                'methods'             => WP_REST_Server::CREATABLE,
                'callback'            => array( $this->backups, 'backup_wiederherstellen' ),
                'permission_callback' => $auth_cb,
                'args'                => array(
                    'backup_id' => array(
                        'required'          => true,
                        'sanitize_callback' => 'sanitize_file_name',
                    ),
                ),
            ) );
        }
    }

    // ── Callbacks ──────────────────────────────────────────────────────────────

    /**
     * GET /discover
     * Gibt die vollstaendige Connector-URL zurueck.
     * Die App muss nur den API-Key kennen und kann die URL automatisch ermitteln.
     * Funktioniert aehnlich wie WP Umbrella: nur Key noetig.
     */
    public function discover( WP_REST_Request $request ) {
        $connector_url = rest_url( WPGRID_API_NS );

        return wpgrid_antwort( true, array(
            'connector_url'     => $connector_url,
            'site_url'          => get_site_url(),
            'site_name'         => get_bloginfo( 'name' ),
            'connector_version' => WPGRID_VERSION,
            'wp_version'        => get_bloginfo( 'version' ),
        ) );
    }

    /**
     * GET /info - Vollstaendige Site-Informationen
     */
    public function site_info( WP_REST_Request $request ) {
        global $wpdb;

        if ( ! function_exists( 'get_plugins' ) ) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }
        if ( ! function_exists( 'wp_get_update_data' ) ) {
            require_once ABSPATH . 'wp-includes/update.php';
        }

        wp_update_plugins();
        wp_update_themes();

        $update_plugins = get_site_transient( 'update_plugins' );
        $alle_plugins   = get_plugins();
        $aktive_plugins = get_option( 'active_plugins', array() );

        $plugin_liste = array();
        foreach ( $alle_plugins as $plugin_datei => $plugin_daten ) {
            $hat_update = isset( $update_plugins->response[ $plugin_datei ] );
            $plugin_liste[] = array(
                'file'             => $plugin_datei,
                'slug'             => dirname( $plugin_datei ),
                'name'             => $plugin_daten['Name'],
                'version'          => $plugin_daten['Version'],
                'author'           => wp_strip_all_tags( $plugin_daten['Author'] ),
                'description'      => wp_trim_words( $plugin_daten['Description'], 20 ),
                'active'           => in_array( $plugin_datei, $aktive_plugins, true ),
                'update_available' => $hat_update,
                'new_version'      => $hat_update ? $update_plugins->response[ $plugin_datei ]->new_version : null,
                'url'              => esc_url( $plugin_daten['PluginURI'] ),
            );
        }

        $theme = wp_get_theme();

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Simple version query, no caching needed
        $mysql_version = $wpdb->get_var( 'SELECT VERSION()' );

        $disk_free  = function_exists( 'disk_free_space' )  ? round( disk_free_space( ABSPATH ) / 1024 / 1024, 2 )  : null;
        $disk_total = function_exists( 'disk_total_space' ) ? round( disk_total_space( ABSPATH ) / 1024 / 1024, 2 ) : null;

        $daten = array(
            'site' => array(
                'url'         => get_site_url(),
                'name'        => get_bloginfo( 'name' ),
                'description' => get_bloginfo( 'description' ),
                'admin_email' => get_option( 'admin_email' ),
                'language'    => get_locale(),
                'timezone'    => wp_timezone_string(),
            ),
            'wordpress' => array(
                'version'    => get_bloginfo( 'version' ),
                'multisite'  => is_multisite(),
                'debug_mode' => defined( 'WP_DEBUG' ) && WP_DEBUG,
            ),
            'server' => array(
                'php_version'     => PHP_VERSION,
                'mysql_version'   => $mysql_version,
                'disk_free_mb'    => $disk_free,
                'disk_total_mb'   => $disk_total,
                'server_software' => isset( $_SERVER['SERVER_SOFTWARE'] ) ? sanitize_text_field( wp_unslash( $_SERVER['SERVER_SOFTWARE'] ) ) : 'unbekannt',
            ),
            'theme' => array(
                'name'       => $theme->get( 'Name' ),
                'version'    => $theme->get( 'Version' ),
                'author'     => $theme->get( 'Author' ),
                'stylesheet' => $theme->get_stylesheet(),
            ),
            'plugins'           => $plugin_liste,
            'plugins_total'     => count( $plugin_liste ),
            'plugins_active'    => count( $aktive_plugins ),
            'plugins_updates'   => isset( $update_plugins->response ) ? count( $update_plugins->response ) : 0,
            'connector_version' => WPGRID_VERSION,
            'timestamp'         => current_time( 'c' ),
        );

        return wpgrid_antwort( true, $daten );
    }

    /**
     * GET /plugins - Plugin-Liste (Kurzversion)
     */
    public function plugins_liste( WP_REST_Request $request ) {
        if ( ! function_exists( 'get_plugins' ) ) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }

        $alle_plugins   = get_plugins();
        $aktive_plugins = get_option( 'active_plugins', array() );
        $update_plugins = get_site_transient( 'update_plugins' );

        $liste = array();
        foreach ( $alle_plugins as $datei => $daten ) {
            $liste[] = array(
                'file'             => $datei,
                'slug'             => dirname( $datei ),
                'name'             => $daten['Name'],
                'version'          => $daten['Version'],
                'active'           => in_array( $datei, $aktive_plugins, true ),
                'update_available' => isset( $update_plugins->response[ $datei ] ),
                'new_version'      => isset( $update_plugins->response[ $datei ] ) ? $update_plugins->response[ $datei ]->new_version : null,
            );
        }

        return wpgrid_antwort( true, array( 'plugins' => $liste, 'total' => count( $liste ) ) );
    }

    /**
     * POST /plugins/activate
     */
    public function plugin_aktivieren( WP_REST_Request $request ) {
        if ( ! function_exists( 'activate_plugin' ) ) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }

        $slug         = $request->get_param( 'slug' );
        $plugin_datei = $this->slug_zu_datei( $slug );

        if ( ! $plugin_datei ) {
            return wpgrid_antwort( false, null, 'Plugin nicht gefunden: ' . esc_html( $slug ) );
        }

        $ergebnis = activate_plugin( $plugin_datei );

        if ( is_wp_error( $ergebnis ) ) {
            return wpgrid_antwort( false, null, $ergebnis->get_error_message() );
        }

        wpgrid_debug( 'Plugin aktiviert', array( 'slug' => $slug ) );
        return wpgrid_antwort( true, array( 'plugin' => $slug, 'status' => 'activated' ) );
    }

    /**
     * POST /plugins/deactivate
     */
    public function plugin_deaktivieren( WP_REST_Request $request ) {
        if ( ! function_exists( 'deactivate_plugins' ) ) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }

        $slug         = $request->get_param( 'slug' );
        $plugin_datei = $this->slug_zu_datei( $slug );

        if ( ! $plugin_datei ) {
            return wpgrid_antwort( false, null, 'Plugin nicht gefunden: ' . esc_html( $slug ) );
        }

        if ( plugin_basename( WPGRID_PLUGIN_FILE ) === $plugin_datei ) {
            return wpgrid_antwort( false, null, 'Der Grid Connector kann nicht remote deaktiviert werden.' );
        }

        deactivate_plugins( $plugin_datei );
        wpgrid_debug( 'Plugin deaktiviert', array( 'slug' => $slug ) );
        return wpgrid_antwort( true, array( 'plugin' => $slug, 'status' => 'deactivated' ) );
    }

    /**
     * POST /plugins/install - Plugin von WordPress.org oder URL installieren
     */
    public function plugin_installieren( WP_REST_Request $request ) {
        require_once ABSPATH . 'wp-admin/includes/plugin.php';
        require_once ABSPATH . 'wp-admin/includes/file.php';
        require_once ABSPATH . 'wp-admin/includes/misc.php';
        require_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';
        require_once ABSPATH . 'wp-admin/includes/class-plugin-upgrader.php';

        $url = $request->get_param( 'url' );

        if ( strpos( $url, 'https://' ) !== 0 ) {
            return wpgrid_antwort( false, null, 'Nur HTTPS-URLs sind erlaubt.' );
        }

        $upgrader = new Plugin_Upgrader( new WP_Ajax_Upgrader_Skin() );
        $ergebnis = $upgrader->install( $url );

        if ( is_wp_error( $ergebnis ) ) {
            return wpgrid_antwort( false, null, $ergebnis->get_error_message() );
        }

        if ( false === $ergebnis ) {
            return wpgrid_antwort( false, null, 'Installation fehlgeschlagen.' );
        }

        return wpgrid_antwort( true, array( 'status' => 'installed', 'url' => $url ) );
    }

    /**
     * GET /report-data - Erweiterte Daten fuer Kundenreports
     */
    public function report_data( WP_REST_Request $request ) {
        global $wpdb;

        if ( ! function_exists( 'get_plugins' ) ) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }

        $alle_plugins   = get_plugins();
        $aktive_plugins = get_option( 'active_plugins', array() );
        $update_plugins = get_site_transient( 'update_plugins' );

        $active_count   = count( $aktive_plugins );
        $inactive_count = count( $alle_plugins ) - $active_count;
        $pending_updates = isset( $update_plugins->response ) ? count( $update_plugins->response ) : 0;

        // DB-Groesse
        $db_size = null;
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.NotPrepared,WordPress.DB.PreparedSQLPlaceholders.UnquotedComplexPlaceholder -- DB name is a trusted constant
        $tables  = $wpdb->get_results( $wpdb->prepare( "SHOW TABLE STATUS FROM `%1s`", DB_NAME ), ARRAY_A );
        if ( $tables ) {
            $total = 0;
            foreach ( $tables as $table ) {
                $total += (float) $table['Data_length'] + (float) $table['Index_length'];
            }
            $db_size = round( $total / 1024 / 1024, 2 ) . ' MB';
        }

        // SSL-Ablaufdatum
        $ssl_expiry = null;
        $site_url   = get_site_url();
        if ( strpos( $site_url, 'https' ) === 0 ) {
            $host = wp_parse_url( $site_url, PHP_URL_HOST );
            if ( $host ) {
                $context = stream_context_create( array( 'ssl' => array( 'capture_peer_cert' => true ) ) );
                $stream  = @stream_socket_client( 'ssl://' . $host . ':443', $errno, $errstr, 5, STREAM_CLIENT_CONNECT, $context );
                if ( $stream ) {
                    $params = stream_context_get_params( $stream );
                    if ( isset( $params['options']['ssl']['peer_certificate'] ) ) {
                        $cert_info = openssl_x509_parse( $params['options']['ssl']['peer_certificate'] );
                        if ( isset( $cert_info['validTo_time_t'] ) ) {
                            $ssl_expiry = gmdate( 'Y-m-d', $cert_info['validTo_time_t'] );
                        }
                    }
                    // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fclose -- stream_socket_client resource
                    fclose( $stream );
                }
            }
        }

        // Letztes Backup
        $last_backup = null;
        $backup_dir  = WP_CONTENT_DIR . '/wpgrid-backups';
        if ( is_dir( $backup_dir ) ) {
            $files = glob( $backup_dir . '/*.zip' );
            if ( $files ) {
                usort( $files, function( $a, $b ) { return filemtime( $b ) - filemtime( $a ); } );
                $last_backup = gmdate( 'Y-m-d H:i', filemtime( $files[0] ) );
            }
        }

        // Admin-User Anzahl
        $admin_users = count_users();
        $admin_count = isset( $admin_users['avail_roles']['administrator'] ) ? $admin_users['avail_roles']['administrator'] : 0;

        // Letzte Anmeldung (via WP)
        $last_login = null;
        $admins     = get_users( array( 'role' => 'administrator', 'number' => 1, 'orderby' => 'ID', 'order' => 'DESC' ) );
        if ( $admins ) {
            $last_login_meta = get_user_meta( $admins[0]->ID, 'last_login', true );
            if ( $last_login_meta ) {
                $last_login = $last_login_meta;
            }
        }

        $daten = array(
            'php_version'      => PHP_VERSION,
            'wp_version'       => get_bloginfo( 'version' ),
            'active_plugins'   => $active_count,
            'inactive_plugins' => $inactive_count,
            'pending_updates'  => $pending_updates,
            'db_size'          => $db_size,
            'last_backup'      => $last_backup,
            'ssl_expiry'       => $ssl_expiry,
            'blocked_logins'   => 0,
            'last_login'       => $last_login,
            'two_fa_active'    => false,
            'admin_users'      => $admin_count,
            'uptime'           => 99.9,
            'resolved_updates' => 0,
            'last_maintenance' => null,
        );

        return wpgrid_antwort( true, $daten );
    }

    /**
     * GET /errors – PHP-Fehleranalyse: debug.log, PHP-Konfiguration, WP-Fehlereinstellungen
     */
    public function fehler_analyse( WP_REST_Request $request ) {
        $errors = array();

        // WP_DEBUG Einstellungen
        $debug_config = array(
            'WP_DEBUG'         => defined( 'WP_DEBUG' ) && WP_DEBUG,
            'WP_DEBUG_LOG'     => defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG,
            'WP_DEBUG_DISPLAY' => defined( 'WP_DEBUG_DISPLAY' ) && WP_DEBUG_DISPLAY,
            'SCRIPT_DEBUG'     => defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG,
        );

        // PHP-Konfiguration
        $php_config = array(
            'version'              => PHP_VERSION,
            'memory_limit'         => ini_get( 'memory_limit' ),
            'max_execution_time'   => ini_get( 'max_execution_time' ),
            'upload_max_filesize'  => ini_get( 'upload_max_filesize' ),
            'post_max_size'        => ini_get( 'post_max_size' ),
            'display_errors'       => ini_get( 'display_errors' ),
            'error_reporting'      => ini_get( 'error_reporting' ),
            'max_input_vars'       => ini_get( 'max_input_vars' ),
        );

        // WP Memory
        $wp_memory = array(
            'wp_memory_limit'     => defined( 'WP_MEMORY_LIMIT' ) ? WP_MEMORY_LIMIT : '40M',
            'wp_max_memory_limit' => defined( 'WP_MAX_MEMORY_LIMIT' ) ? WP_MAX_MEMORY_LIMIT : '256M',
            'current_usage_mb'    => round( memory_get_usage() / 1024 / 1024, 2 ),
            'peak_usage_mb'       => round( memory_get_peak_usage() / 1024 / 1024, 2 ),
        );

        // debug.log lesen (letzte 100 Zeilen)
        $debug_log = array();
        $log_path = WP_CONTENT_DIR . '/debug.log';
        if ( file_exists( $log_path ) ) {
            global $wp_filesystem;
            if ( empty( $wp_filesystem ) ) {
                require_once ABSPATH . 'wp-admin/includes/file.php';
                WP_Filesystem();
            }
            $raw_log = $wp_filesystem->get_contents( $log_path );
            $lines   = $raw_log ? array_slice( explode( "\n", $raw_log ), -100 ) : array();

            $fatal_count = 0;
            $warning_count = 0;
            $notice_count = 0;
            $deprecated_count = 0;
            foreach ( $lines as $line ) {
                if ( stripos( $line, 'fatal error' ) !== false ) $fatal_count++;
                elseif ( stripos( $line, 'warning' ) !== false ) $warning_count++;
                elseif ( stripos( $line, 'notice' ) !== false ) $notice_count++;
                elseif ( stripos( $line, 'deprecated' ) !== false ) $deprecated_count++;
            }

            $debug_log = array(
                'exists'           => true,
                'size_kb'          => round( $log_size / 1024, 1 ),
                'last_100_lines'   => $lines,
                'fatal_count'      => $fatal_count,
                'warning_count'    => $warning_count,
                'notice_count'     => $notice_count,
                'deprecated_count' => $deprecated_count,
            );
        } else {
            $debug_log = array( 'exists' => false );
        }

        // Fehlende PHP-Extensions prüfen
        $required_extensions = array( 'curl', 'mbstring', 'openssl', 'xml', 'zip', 'gd', 'intl', 'json' );
        $missing_extensions = array();
        foreach ( $required_extensions as $ext ) {
            if ( ! extension_loaded( $ext ) ) {
                $missing_extensions[] = $ext;
            }
        }

        return wpgrid_antwort( true, array(
            'debug_config'       => $debug_config,
            'php_config'         => $php_config,
            'wp_memory'          => $wp_memory,
            'debug_log'          => $debug_log,
            'missing_extensions' => $missing_extensions,
            'timestamp'          => current_time( 'c' ),
        ) );
    }

    /**
     * POST /auto-login – Erstellt einen einmaligen Login-Token (60 Sek. gültig)
     */
    public function auto_login_token_erstellen( WP_REST_Request $request ) {
        $token = bin2hex( random_bytes( 32 ) );
        $admin = get_users( array( 'role' => 'administrator', 'number' => 1, 'orderby' => 'ID', 'order' => 'ASC' ) );
        if ( empty( $admin ) ) {
            return wpgrid_antwort( false, null, 'Kein Administrator-Konto gefunden.' );
        }

        set_transient( 'wpgrid_autologin_' . $token, array(
            'user_id'    => $admin[0]->ID,
            'created_at' => time(),
        ), 60 );

        $login_url = rest_url( WPGRID_API_NS . '/auto-login/' . $token );

        return wpgrid_antwort( true, array( 'login_url' => $login_url, 'expires_in' => 60 ) );
    }

    /**
     * GET /auto-login/{token} – Führt den einmaligen Login aus und leitet zum wp-admin weiter
     */
    public function auto_login_ausfuehren( WP_REST_Request $request ) {
        $token = sanitize_text_field( $request->get_param( 'token' ) );
        $data  = get_transient( 'wpgrid_autologin_' . $token );

        if ( ! $data || ! isset( $data['user_id'] ) ) {
            wp_die(
                'Dieser Login-Link ist ungültig oder abgelaufen.',
                'WP-Grid Auto-Login',
                array( 'response' => 403 )
            );
        }

        delete_transient( 'wpgrid_autologin_' . $token );

        $user = get_user_by( 'ID', $data['user_id'] );
        if ( ! $user ) {
            wp_die( 'Benutzer nicht gefunden.', 'WP-Grid Auto-Login', array( 'response' => 404 ) );
        }

        wp_set_current_user( $user->ID );
        wp_set_auth_cookie( $user->ID, false );
        // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound -- wp_login is a core WordPress hook
        do_action( 'wp_login', $user->user_login, $user );

        wpgrid_debug( 'Auto-Login durchgefuehrt', array( 'user_id' => $user->ID ) );

        wp_safe_redirect( admin_url() );
        exit;
    }

    /**
     * POST /config/set – wp-config.php Konstanten setzen/ändern
     */
    public function config_setzen( WP_REST_Request $request ) {
        $erlaubte_keys = array(
            'WP_DEBUG', 'WP_DEBUG_LOG', 'WP_DEBUG_DISPLAY', 'SCRIPT_DEBUG',
            'WP_MEMORY_LIMIT', 'WP_MAX_MEMORY_LIMIT',
            'WP_POST_REVISIONS', 'AUTOSAVE_INTERVAL', 'EMPTY_TRASH_DAYS',
            'WP_CACHE', 'CONCATENATE_SCRIPTS', 'COMPRESS_SCRIPTS', 'COMPRESS_CSS',
            'DISALLOW_FILE_EDIT', 'DISALLOW_FILE_MODS',
            'WP_AUTO_UPDATE_CORE',
        );

        $settings = $request->get_param( 'settings' );
        if ( ! is_array( $settings ) || empty( $settings ) ) {
            return wpgrid_antwort( false, null, 'Keine Einstellungen übergeben.' );
        }

        $config_path = ABSPATH . 'wp-config.php';
        if ( ! file_exists( $config_path ) || ! wp_is_writable( $config_path ) ) {
            return wpgrid_antwort( false, null, 'wp-config.php nicht beschreibbar.' );
        }

        $content = file_get_contents( $config_path );
        if ( false === $content ) {
            return wpgrid_antwort( false, null, 'wp-config.php konnte nicht gelesen werden.' );
        }

        $geaendert = array();

        foreach ( $settings as $key => $value ) {
            $key = strtoupper( sanitize_text_field( $key ) );
            if ( ! in_array( $key, $erlaubte_keys, true ) ) {
                continue;
            }

            if ( is_bool( $value ) || $value === 'true' || $value === 'false' ) {
                $php_value = ( $value === true || $value === 'true' ) ? 'true' : 'false';
            } elseif ( is_numeric( $value ) ) {
                $php_value = intval( $value );
            } else {
                $php_value = "'" . addslashes( sanitize_text_field( $value ) ) . "'";
            }

            $pattern = "/define\s*\(\s*['\"]" . preg_quote( $key, '/' ) . "['\"]\s*,\s*[^)]+\)\s*;/";

            if ( is_bool( $value ) || $value === 'true' || $value === 'false' || is_numeric( $value ) ) {
                $replacement = "define( '{$key}', {$php_value} );";
            } else {
                $replacement = "define( '{$key}', {$php_value} );";
            }

            if ( preg_match( $pattern, $content ) ) {
                $content = preg_replace( $pattern, $replacement, $content );
            } else {
                $marker = "/* That's all, stop editing!";
                $alt_marker = "/** Absolute path to the WordPress directory.";
                if ( strpos( $content, $marker ) !== false ) {
                    $content = str_replace( $marker, $replacement . "\n" . $marker, $content );
                } elseif ( strpos( $content, $alt_marker ) !== false ) {
                    $content = str_replace( $alt_marker, $replacement . "\n" . $alt_marker, $content );
                } else {
                    $content .= "\n" . $replacement . "\n";
                }
            }

            $geaendert[] = $key;
        }

        if ( empty( $geaendert ) ) {
            return wpgrid_antwort( false, null, 'Keine gültigen Einstellungen.' );
        }

        $backup_path = ABSPATH . 'wp-config.php.wpgrid-backup';
        copy( $config_path, $backup_path );

        if ( false === file_put_contents( $config_path, $content ) ) {
            if ( file_exists( $backup_path ) ) {
                copy( $backup_path, $config_path );
            }
            return wpgrid_antwort( false, null, 'Schreiben fehlgeschlagen.' );
        }

        wpgrid_debug( 'wp-config.php geändert', array( 'keys' => $geaendert ) );
        return wpgrid_antwort( true, array( 'changed' => $geaendert ) );
    }

    /**
     * GET /htaccess – .htaccess Inhalt lesen
     */
    public function htaccess_lesen( WP_REST_Request $request ) {
        $path = ABSPATH . '.htaccess';

        if ( ! file_exists( $path ) ) {
            return wpgrid_antwort( true, array(
                'exists'  => false,
                'content' => '',
                'size_kb' => 0,
            ) );
        }

        $content = file_get_contents( $path );
        return wpgrid_antwort( true, array(
            'exists'    => true,
            'content'   => $content,
            'size_kb'   => round( strlen( $content ) / 1024, 1 ),
            'writable'  => wp_is_writable( $path ),
            'modified'  => gmdate( 'Y-m-d H:i:s', filemtime( $path ) ),
        ) );
    }

    /**
     * POST /htaccess – .htaccess Inhalt schreiben
     */
    public function htaccess_schreiben( WP_REST_Request $request ) {
        $path    = ABSPATH . '.htaccess';
        $content = $request->get_param( 'content' );

        if ( $content === null ) {
            return wpgrid_antwort( false, null, 'Kein Inhalt übergeben.' );
        }

        if ( file_exists( $path ) ) {
            $backup = ABSPATH . '.htaccess.wpgrid-backup';
            copy( $path, $backup );
        }

        if ( false === file_put_contents( $path, $content ) ) {
            return wpgrid_antwort( false, null, '.htaccess konnte nicht geschrieben werden.' );
        }

        wpgrid_debug( '.htaccess aktualisiert' );
        return wpgrid_antwort( true, array( 'status' => 'saved' ) );
    }

    /**
     * GET /core-info – WP Core Version + Update-Status
     */
    public function core_info( WP_REST_Request $request ) {
        require_once ABSPATH . 'wp-admin/includes/update.php';

        wp_version_check();
        $update_core = get_site_transient( 'update_core' );

        $current_version = get_bloginfo( 'version' );
        $update_available = false;
        $new_version = null;

        if ( isset( $update_core->updates ) ) {
            foreach ( $update_core->updates as $u ) {
                if ( 'upgrade' === $u->response ) {
                    $update_available = true;
                    $new_version = $u->version;
                    break;
                }
            }
        }

        return wpgrid_antwort( true, array(
            'version'          => $current_version,
            'update_available' => $update_available,
            'new_version'      => $new_version,
            'php_version'      => PHP_VERSION,
            'db_version'       => get_option( 'db_version' ),
        ) );
    }

    // ── Hilfsmethoden ──────────────────────────────────────────────────────────

    /**
     * Slug (z.B. "woocommerce") zu Plugin-Datei (z.B. "woocommerce/woocommerce.php") aufloesen
     */
    private function slug_zu_datei( $slug ) {
        if ( ! function_exists( 'get_plugins' ) ) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }

        $slug = sanitize_text_field( $slug );

        if ( strpos( $slug, '/' ) !== false ) {
            $alle = get_plugins();
            return isset( $alle[ $slug ] ) ? $slug : null;
        }

        foreach ( array_keys( get_plugins() ) as $datei ) {
            if ( dirname( $datei ) === $slug || $datei === $slug . '.php' ) {
                return $datei;
            }
        }

        return null;
    }

    /**
     * Validiert einen Plugin/Theme-Slug (nur alphanumerisch + Bindestrich/Slash/Punkt)
     */
    public function slug_validieren( $wert ) {
        return (bool) preg_match( '/^[a-z0-9\-_\/\.]+$/i', $wert );
    }

    /**
     * Validiert eine URL
     */
    public function url_validieren( $wert ) {
        return filter_var( $wert, FILTER_VALIDATE_URL ) !== false;
    }
}
