<?php
/**
 * API-Key Authentifizierung und Rate Limiting
 *
 * @package WP_Grid_Connector
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class WPGrid_Auth {

    /**
     * Prueft ob der API-Key im Request gueltig ist.
     * Wird als permission_callback fuer alle REST-Routen verwendet.
     */
    public function api_key_pruefen( WP_REST_Request $request ) {
        $start = microtime( true );

        // API-Key aus Header oder Query-Parameter lesen
        $eingabe_key = $request->get_header( 'X-WPGrid-Key' );
        if ( ! $eingabe_key ) {
            $eingabe_key = $request->get_param( 'api_key' );
        }

        if ( ! $eingabe_key ) {
            $this->log_schreiben( $request, 401, microtime( true ) - $start, 'Kein API-Key angegeben' );
            return new WP_Error(
                'wpgrid_no_key',
                __( 'API-Key fehlt. Bitte X-WPGrid-Key Header oder api_key Parameter angeben.', 'grid-connector' ),
                array( 'status' => 401 )
            );
        }

        // Rate Limiting pruefen
        if ( $this->rate_limit_ueberschritten( $eingabe_key ) ) {
            $this->log_schreiben( $request, 429, microtime( true ) - $start, 'Rate Limit ueberschritten' );
            return new WP_Error(
                'wpgrid_rate_limit',
                /* translators: %d: max requests per minute */ sprintf( __( 'Too many requests. Maximum %d requests per minute.', 'grid-connector' ), WPGRID_RATE_LIMIT ),
                array( 'status' => 429 )
            );
        }

        // Gespeicherten Key laden und sicher vergleichen
        $gespeicherter_key = get_option( WPGRID_OPTION_KEY, '' );
        if ( ! hash_equals( $gespeicherter_key, $eingabe_key ) ) {
            $this->log_schreiben( $request, 403, microtime( true ) - $start, 'Ungültiger API-Key' );
            return new WP_Error(
                'wpgrid_invalid_key',
                __( 'Ungültiger API-Key.', 'grid-connector' ),
                array( 'status' => 403 )
            );
        }

        // Erfolgreichen Request loggen
        $this->log_schreiben( $request, 200, microtime( true ) - $start );
        return true;
    }

    /**
     * Prueft ob das Rate Limit fuer diesen API-Key ueberschritten wurde.
     * Nutzt WordPress Transients als einfachen Zaehler.
     */
    private function rate_limit_ueberschritten( $api_key ) {
        $transient_key = 'wpgrid_rl_' . substr( md5( $api_key ), 0, 16 );
        $zaehler       = (int) get_transient( $transient_key );

        if ( $zaehler >= WPGRID_RATE_LIMIT ) {
            return true;
        }

        if ( 0 === $zaehler ) {
            set_transient( $transient_key, 1, 60 );
        } else {
            set_transient( $transient_key, $zaehler + 1, 60 );
        }

        return false;
    }

    /**
     * Ermittelt die echte Client-IP (auch hinter Proxies)
     */
    public function client_ip_ermitteln() {
        $header_reihenfolge = array(
            'HTTP_CF_CONNECTING_IP', // Cloudflare
            'HTTP_X_FORWARDED_FOR',
            'HTTP_X_REAL_IP',
            'REMOTE_ADDR',
        );

        foreach ( $header_reihenfolge as $header ) {
            if ( ! empty( $_SERVER[ $header ] ) ) {
                $ip = sanitize_text_field( wp_unslash( $_SERVER[ $header ] ) );
                if ( strpos( $ip, ',' ) !== false ) {
                    $ip = trim( explode( ',', $ip )[0] );
                }
                if ( filter_var( $ip, FILTER_VALIDATE_IP ) ) {
                    return $ip;
                }
            }
        }

        return '0.0.0.0';
    }

    /**
     * Request in die Log-Tabelle schreiben
     */
    private function log_schreiben( WP_REST_Request $request, $status_code, $antwortzeit, $fehler = '' ) {
        wpgrid_log_schreiben(
            $request->get_method(),
            $request->get_route(),
            $this->client_ip_ermitteln(),
            $status_code,
            round( $antwortzeit * 1000, 2 ),
            $fehler
        );
    }
}