<?php
/**
 * Backup-Verwaltung: Erstellen, Auflisten, Löschen und Wiederherstellen
 *
 * @package WP_Grid_Connector
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class WPGrid_Backups {

    private $backup_dir;

    public function __construct() {
        $this->backup_dir = WP_CONTENT_DIR . '/wpgrid-backups';
    }

    /**
     * GET /backups – alle Backups auflisten
     */
    public function backups_listen( WP_REST_Request $request ) {
        $this->verzeichnis_sichern();

        $dateien = glob( $this->backup_dir . '/wpgrid-backup-*.zip' );
        if ( ! $dateien ) {
            return wpgrid_antwort( true, array( 'backups' => array() ) );
        }

        // Neueste zuerst
        usort( $dateien, function( $a, $b ) {
            return filemtime( $b ) - filemtime( $a );
        } );

        $liste = array();
        foreach ( $dateien as $datei ) {
            $name      = basename( $datei );
            $groesse   = filesize( $datei );
            $erstellt  = filemtime( $datei );

            // ID aus Dateiname extrahieren: wpgrid-backup-{timestamp}-{type}.zip
            preg_match( '/wpgrid-backup-(\d+)-(\w+)\.zip/', $name, $treffer );
            $timestamp = isset( $treffer[1] ) ? (int) $treffer[1] : $erstellt;
            $typ       = isset( $treffer[2] ) ? $treffer[2] : 'manual';

            $liste[] = array(
                'id'         => $name,
                'label'      => $this->backup_label( $typ, $timestamp ),
                'filename'   => $name,
                'createdAt'  => gmdate( 'c', $timestamp ),
                'size'       => $this->groesse_formatieren( $groesse ),
                'type'       => $typ,
                'status'     => 'success',
            );
        }

        return wpgrid_antwort( true, array( 'backups' => $liste ) );
    }

    /**
     * POST /backups – neues Backup erstellen
     */
    public function backup_erstellen( WP_REST_Request $request ) {
        $this->verzeichnis_sichern();

        $typ       = sanitize_text_field( $request->get_param( 'type' ) ?: 'manual' );
        $timestamp = time();
        $dateiname = "wpgrid-backup-{$timestamp}-{$typ}.zip";
        $pfad      = $this->backup_dir . '/' . $dateiname;

        if ( ! class_exists( 'ZipArchive' ) ) {
            return wpgrid_antwort( false, null, 'ZipArchive ist auf diesem Server nicht verfügbar.' );
        }

        $zip = new ZipArchive();
        if ( $zip->open( $pfad, ZipArchive::CREATE ) !== true ) {
            return wpgrid_antwort( false, null, 'Backup-Datei konnte nicht erstellt werden.' );
        }

        // WordPress-Kern-Dateien sichern (wp-content)
        $wp_content = WP_CONTENT_DIR;
        $this->verzeichnis_zu_zip_hinzufuegen( $zip, $wp_content, 'wp-content' );

        // Datenbank-Export als SQL hinzufügen
        $sql = $this->datenbank_exportieren();
        if ( $sql ) {
            $zip->addFromString( 'database.sql', $sql );
        }

        $zip->close();

        // Alte Backups bereinigen (max. 10 behalten)
        $this->alte_backups_bereinigen( 10 );

        $groesse = filesize( $pfad );

        wpgrid_debug( 'Backup erstellt', array( 'datei' => $dateiname, 'groesse' => $groesse ) );

        return wpgrid_antwort( true, array(
            'id'        => $dateiname,
            'label'     => $this->backup_label( $typ, $timestamp ),
            'filename'  => $dateiname,
            'createdAt' => gmdate( 'c', $timestamp ),
            'size'      => $this->groesse_formatieren( $groesse ),
            'type'      => $typ,
            'status'    => 'success',
        ) );
    }

    /**
     * DELETE /backups/{id} – Backup löschen
     */
    public function backup_loeschen( WP_REST_Request $request ) {
        $backup_id = sanitize_file_name( $request->get_param( 'backup_id' ) );
        $pfad      = $this->backup_dir . '/' . $backup_id;

        if ( ! file_exists( $pfad ) || ! $this->pfad_erlaubt( $pfad ) ) {
            return wpgrid_antwort( false, null, 'Backup nicht gefunden.' );
        }

        wp_delete_file( $pfad );
        if ( file_exists( $pfad ) ) {
            return wpgrid_antwort( false, null, 'Backup konnte nicht gelöscht werden.' );
        }

        wpgrid_debug( 'Backup gelöscht', array( 'datei' => $backup_id ) );
        return wpgrid_antwort( true, array( 'deleted' => $backup_id ) );
    }

    /**
     * POST /backups/{id}/restore – Backup einspielen (Rollback)
     */
    public function backup_wiederherstellen( WP_REST_Request $request ) {
        $backup_id = sanitize_file_name( $request->get_param( 'backup_id' ) );
        $pfad      = $this->backup_dir . '/' . $backup_id;

        if ( ! file_exists( $pfad ) || ! $this->pfad_erlaubt( $pfad ) ) {
            return wpgrid_antwort( false, null, 'Backup nicht gefunden.' );
        }

        if ( ! class_exists( 'ZipArchive' ) ) {
            return wpgrid_antwort( false, null, 'ZipArchive ist auf diesem Server nicht verfügbar.' );
        }

        $zip = new ZipArchive();
        if ( $zip->open( $pfad ) !== true ) {
            return wpgrid_antwort( false, null, 'Backup-Datei konnte nicht geöffnet werden.' );
        }

        // wp-content wiederherstellen
        $zip->extractTo( dirname( WP_CONTENT_DIR ) );
        $zip->close();

        // Datenbank wiederherstellen wenn vorhanden
        $sql_pfad = $this->backup_dir . '/temp_restore.sql';
        if ( $zip->open( $pfad ) === true ) {
            $sql_inhalt = $zip->getFromName( 'database.sql' );
            $zip->close();
            if ( $sql_inhalt ) {
                file_put_contents( $sql_pfad, $sql_inhalt );
                $this->datenbank_importieren( $sql_pfad );
                wp_delete_file( $sql_pfad );
            }
        }

        wp_cache_flush();
        wpgrid_debug( 'Backup wiederhergestellt', array( 'datei' => $backup_id ) );

        return wpgrid_antwort( true, array(
            'restored'  => $backup_id,
            'nachricht' => 'Backup erfolgreich eingespielt.',
        ) );
    }

    // ── Hilfsmethoden ──────────────────────────────────────────────────────────

    private function verzeichnis_sichern() {
        if ( ! file_exists( $this->backup_dir ) ) {
            wp_mkdir_p( $this->backup_dir );
            // .htaccess: direkten Zugriff sperren
            file_put_contents( $this->backup_dir . '/.htaccess', "Order deny,allow\nDeny from all\n" );
            // index.php: leere Datei als Schutz
            file_put_contents( $this->backup_dir . '/index.php', '<?php // Silence is golden.' );
        }
    }

    private function verzeichnis_zu_zip_hinzufuegen( ZipArchive $zip, string $verzeichnis, string $basis ) {
        $dateien = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator( $verzeichnis, RecursiveDirectoryIterator::SKIP_DOTS ),
            RecursiveIteratorIterator::LEAVES_ONLY
        );

        // Backup-Verzeichnis selbst überspringen
        $backup_rel = str_replace( WP_CONTENT_DIR, '', $this->backup_dir );

        foreach ( $dateien as $datei ) {
            if ( $datei->isDir() ) {
                continue;
            }

            $datei_pfad = $datei->getRealPath();

            // Eigene Backups nicht in das Backup packen
            if ( strpos( $datei_pfad, $this->backup_dir ) !== false ) {
                continue;
            }

            $relativer_pfad = $basis . '/' . substr( $datei_pfad, strlen( $verzeichnis ) + 1 );
            $zip->addFile( $datei_pfad, $relativer_pfad );
        }
    }

    private function datenbank_exportieren(): string {
        global $wpdb;

        $sql    = '';
        $tables = $wpdb->get_col( 'SHOW TABLES' ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching

        foreach ( $tables as $tabelle ) {
            $create = $wpdb->get_row( "SHOW CREATE TABLE `{$tabelle}`", ARRAY_N ); // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared,WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.DirectDatabaseQuery.SchemaChange,PluginCheck.Security.DirectDB.UnescapedDBParameter
            if ( $create ) {
                $sql .= "\n\n-- Tabelle: {$tabelle}\n";
                $sql .= "DROP TABLE IF EXISTS `{$tabelle}`;\n";
                $sql .= $create[1] . ";\n\n";
            }

            $zeilen = $wpdb->get_results( "SELECT * FROM `{$tabelle}`", ARRAY_A ); // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared,WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,PluginCheck.Security.DirectDB.UnescapedDBParameter
            foreach ( $zeilen as $zeile ) {
                $werte = array_map( function( $wert ) use ( $wpdb ) {
                    if ( $wert === null ) return 'NULL';
                    return "'" . esc_sql( $wert ) . "'";
                }, $zeile );

                $sql .= "INSERT INTO `{$tabelle}` VALUES (" . implode( ', ', $werte ) . ");\n";
            }
        }

        return $sql;
    }

    private function datenbank_importieren( string $sql_pfad ) {
        global $wpdb;

        $sql     = file_get_contents( $sql_pfad );
        $queries = explode( ";\n", $sql );

        foreach ( $queries as $query ) {
            $query = trim( $query );
            if ( ! empty( $query ) ) {
                $wpdb->query( $query ); // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared,WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,PluginCheck.Security.DirectDB.UnescapedDBParameter
            }
        }
    }

    private function alte_backups_bereinigen( int $max ) {
        $dateien = glob( $this->backup_dir . '/wpgrid-backup-*.zip' );
        if ( ! $dateien || count( $dateien ) <= $max ) {
            return;
        }

        usort( $dateien, function( $a, $b ) {
            return filemtime( $b ) - filemtime( $a );
        } );

        $zu_loeschen = array_slice( $dateien, $max );
        foreach ( $zu_loeschen as $datei ) {
            wp_delete_file( $datei );
        }
    }

    private function pfad_erlaubt( string $pfad ): bool {
        $real = realpath( $pfad );
        $dir  = realpath( $this->backup_dir );
        return $real && $dir && strpos( $real, $dir ) === 0;
    }

    private function groesse_formatieren( int $bytes ): string {
        if ( $bytes >= 1073741824 ) return round( $bytes / 1073741824, 2 ) . ' GB';
        if ( $bytes >= 1048576 )   return round( $bytes / 1048576, 2 ) . ' MB';
        if ( $bytes >= 1024 )      return round( $bytes / 1024, 2 ) . ' KB';
        return $bytes . ' B';
    }

    private function backup_label( string $typ, int $timestamp ): string {
        $datum = wp_date( 'd.m.Y H:i', $timestamp );
        return $typ === 'manual' ? "Manuell – {$datum}" : "Automatisch – {$datum}";
    }
}
