<?php
/**
 * File Manager - sichere Dateioperationen ueber WordPress Filesystem API
 *
 * @package WP_Grid_Connector
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class WPGrid_Files {

    /**
     * Erlaubte Basis-Verzeichnisse (Directory Traversal verhindern)
     */
    private $erlaubte_basis_verzeichnisse;

    public function __construct() {
        $upload_dir = wp_upload_dir();
        $this->erlaubte_basis_verzeichnisse = array(
            realpath( ABSPATH ),
            realpath( WP_CONTENT_DIR ),
            realpath( WP_CONTENT_DIR . '/uploads' ),
            realpath( WP_CONTENT_DIR . '/themes' ),
            realpath( WP_CONTENT_DIR . '/plugins' ),
            realpath( $upload_dir['basedir'] ),
        );
    }

    /**
     * GET /files/list - Verzeichnis-Inhalt auflisten
     */
    public function dateien_listen( WP_REST_Request $request ) {
        $pfad_param = $request->get_param( 'path' );
        $absoluter_pfad = $this->pfad_aufloesen( $pfad_param );

        if ( is_wp_error( $absoluter_pfad ) ) {
            return wpgrid_antwort( false, null, $absoluter_pfad->get_error_message() );
        }

        if ( ! is_dir( $absoluter_pfad ) ) {
            return wpgrid_antwort( false, null, 'Verzeichnis nicht gefunden: ' . esc_html( $pfad_param ) );
        }

        $eintraege = array();
        $iterator  = new DirectoryIterator( $absoluter_pfad );

        foreach ( $iterator as $eintrag ) {
            if ( $eintrag->isDot() ) {
                continue;
            }

            $eintraege[] = array(
                'name'          => $eintrag->getFilename(),
                'typ'           => $eintrag->isDir() ? 'verzeichnis' : 'datei',
                'groesse_bytes' => $eintrag->isFile() ? $eintrag->getSize() : null,
                'geaendert'     => gmdate( 'Y-m-d H:i:s', $eintrag->getMTime() ),
                'berechtigungen' => substr( sprintf( '%o', $eintrag->getPerms() ), -4 ),
                'pfad'          => trailingslashit( $pfad_param ) . $eintrag->getFilename(),
            );
        }

        usort( $eintraege, function ( $a, $b ) {
            if ( $a['typ'] !== $b['typ'] ) {
                return $a['typ'] === 'verzeichnis' ? -1 : 1;
            }
            return strcmp( $a['name'], $b['name'] );
        } );

        return wpgrid_antwort( true, array(
            'pfad'    => $pfad_param,
            'eintraege' => $eintraege,
            'anzahl'  => count( $eintraege ),
        ) );
    }

    /**
     * POST /files/upload - Datei hochladen
     */
    public function datei_hochladen( WP_REST_Request $request ) {
        $dateien = $request->get_file_params();

        if ( empty( $dateien['file'] ) ) {
            return wpgrid_antwort( false, null, 'Keine Datei im Request gefunden (Feld: file).' );
        }

        $ziel_pfad_param = $request->get_param( 'path' );
        if ( ! $ziel_pfad_param ) {
            $upload_dir      = wp_upload_dir();
            $ziel_pfad_param = str_replace( ABSPATH, '/', $upload_dir['path'] );
        }

        $absoluter_ziel_pfad = $this->pfad_aufloesen( $ziel_pfad_param );
        if ( is_wp_error( $absoluter_ziel_pfad ) ) {
            return wpgrid_antwort( false, null, $absoluter_ziel_pfad->get_error_message() );
        }

        $datei      = $dateien['file'];
        $dateiname  = sanitize_file_name( $datei['name'] );
        $ziel_datei = trailingslashit( $absoluter_ziel_pfad ) . $dateiname;

        // WordPress Filesystem API verwenden
        global $wp_filesystem;
        if ( ! $this->filesystem_initialisieren() ) {
            return wpgrid_antwort( false, null, 'WordPress Filesystem konnte nicht initialisiert werden.' );
        }

        if ( ! $wp_filesystem->move( $datei['tmp_name'], $ziel_datei, true ) ) {
            return wpgrid_antwort( false, null, 'Datei konnte nicht gespeichert werden.' );
        }

        wpgrid_debug( 'Datei hochgeladen', array( 'ziel' => $ziel_datei ) );
        return wpgrid_antwort( true, array(
            'dateiname' => $dateiname,
            'pfad'      => trailingslashit( $ziel_pfad_param ) . $dateiname,
            'groesse'   => $datei['size'],
        ) );
    }

    /**
     * DELETE /files/delete - Datei loeschen
     */
    public function datei_loeschen( WP_REST_Request $request ) {
        $pfad_param     = $request->get_param( 'path' );
        $absoluter_pfad = $this->pfad_aufloesen( $pfad_param );

        if ( is_wp_error( $absoluter_pfad ) ) {
            return wpgrid_antwort( false, null, $absoluter_pfad->get_error_message() );
        }

        if ( ! file_exists( $absoluter_pfad ) ) {
            return wpgrid_antwort( false, null, 'Datei nicht gefunden: ' . esc_html( $pfad_param ) );
        }

        // Verzeichnisse nicht loeschen (nur Dateien)
        if ( is_dir( $absoluter_pfad ) ) {
            return wpgrid_antwort( false, null, 'Verzeichnisse koennen nicht ueber diesen Endpoint geloescht werden.' );
        }

        global $wp_filesystem;
        if ( ! $this->filesystem_initialisieren() ) {
            return wpgrid_antwort( false, null, 'WordPress Filesystem konnte nicht initialisiert werden.' );
        }

        if ( ! $wp_filesystem->delete( $absoluter_pfad ) ) {
            return wpgrid_antwort( false, null, 'Datei konnte nicht geloescht werden.' );
        }

        wpgrid_debug( 'Datei geloescht', array( 'pfad' => $pfad_param ) );
        return wpgrid_antwort( true, array( 'geloescht' => $pfad_param ) );
    }

    /**
     * GET /files/download - Datei herunterladen
     */
    public function datei_herunterladen( WP_REST_Request $request ) {
        $pfad_param     = $request->get_param( 'path' );
        $absoluter_pfad = $this->pfad_aufloesen( $pfad_param );

        if ( is_wp_error( $absoluter_pfad ) ) {
            return wpgrid_antwort( false, null, $absoluter_pfad->get_error_message() );
        }

        if ( ! is_file( $absoluter_pfad ) ) {
            return wpgrid_antwort( false, null, 'Datei nicht gefunden.' );
        }

        $dateiname = basename( $absoluter_pfad );
        $mime_typ  = mime_content_type( $absoluter_pfad );
        if ( ! $mime_typ ) {
            $mime_typ = 'application/octet-stream';
        }

        // Datei-Inhalt als Base64 zurueckgeben (sicher fuer JSON-Transport)
        // phpcs:ignore WordPress.WP.AlternativeFunctions.file_get_contents_file_get_contents
        $inhalt = file_get_contents( $absoluter_pfad );
        if ( false === $inhalt ) {
            return wpgrid_antwort( false, null, 'Datei konnte nicht gelesen werden.' );
        }

        return wpgrid_antwort( true, array(
            'dateiname' => $dateiname,
            'mime_typ'  => $mime_typ,
            'groesse'   => filesize( $absoluter_pfad ),
            'inhalt'    => base64_encode( $inhalt ),
        ) );
    }

    // â”€â”€ Hilfsmethoden â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€

    /**
     * Pfad-Parameter in absoluten, validierten Pfad umwandeln.
     * Verhindert Directory Traversal.
     */
    private function pfad_aufloesen( $pfad_param ) {
        // Relativen Pfad zu absolutem machen
        if ( strpos( $pfad_param, ABSPATH ) !== 0 ) {
            $absoluter_pfad = realpath( ABSPATH . ltrim( $pfad_param, '/' ) );
        } else {
            $absoluter_pfad = realpath( $pfad_param );
        }

        if ( false === $absoluter_pfad ) {
            // Pfad existiert noch nicht (z.B. Upload-Ziel) - Elternverzeichnis pruefen
            $eltern = realpath( dirname( ABSPATH . ltrim( $pfad_param, '/' ) ) );
            if ( false === $eltern ) {
                return new WP_Error( 'wpgrid_invalid_path', 'UngÃ¼ltiger Pfad: ' . esc_html( $pfad_param ) );
            }
            $absoluter_pfad = $eltern . DIRECTORY_SEPARATOR . basename( $pfad_param );
        }

        // Sicherstellen dass der Pfad innerhalb erlaubter Verzeichnisse liegt
        foreach ( $this->erlaubte_basis_verzeichnisse as $basis ) {
            if ( $basis && strpos( $absoluter_pfad, $basis ) === 0 ) {
                return $absoluter_pfad;
            }
        }

        return new WP_Error(
            'wpgrid_path_not_allowed',
            'Zugriff auf diesen Pfad ist nicht erlaubt: ' . esc_html( $pfad_param )
        );
    }

    /**
     * WordPress Filesystem API initialisieren
     */
    private function filesystem_initialisieren() {
        global $wp_filesystem;

        if ( ! function_exists( 'WP_Filesystem' ) ) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
        }

        return WP_Filesystem();
    }
}
