<?php
/**
 * Health-Check Klasse
 *
 * @package WP_Grid_Connector
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class WPGrid_Health {

    /**
     * GET /health - Umfassender Gesundheitscheck der WordPress-Installation
     */
    public function health_check( WP_REST_Request $request ) {
        $probleme = array();
        $warnungen = array();

        $ergebnisse = array(
            'ssl'              => $this->ssl_pruefen( $probleme, $warnungen ),
            'php'              => $this->php_pruefen( $warnungen ),
            'plugins'          => $this->veraltete_plugins_zaehlen(),
            'datenbank'        => $this->datenbank_groesse(),
            'uploads'          => $this->uploads_groesse(),
            'dateiberechtigungen' => $this->dateiberechtigungen_pruefen( $probleme ),
            'debug_modus'      => $this->debug_modus_pruefen( $warnungen ),
            'wp_version'       => $this->wp_version_pruefen( $warnungen ),
            'server'           => $this->server_info(),
            'wp_environment'   => $this->wp_environment_info(),
            'theme'            => $this->theme_info(),
        );

        $gesamt_status = empty( $probleme ) ? ( empty( $warnungen ) ? 'gut' : 'warnung' ) : 'kritisch';

        return wpgrid_antwort( true, array(
            'status'     => $gesamt_status,
            'probleme'   => $probleme,
            'warnungen'  => $warnungen,
            'details'    => $ergebnisse,
            'timestamp'  => current_time( 'c' ),
        ) );
    }

    /**
     * SSL-Zertifikat pruefen
     */
    private function ssl_pruefen( &$probleme, &$warnungen ) {
        $site_url = get_site_url();

        if ( strpos( $site_url, 'https://' ) === false ) {
            $probleme[] = 'Kein SSL: Website nutzt kein HTTPS.';
            return array( 'aktiv' => false, 'ablauf' => null, 'tage_bis_ablauf' => null );
        }

        $host = wp_parse_url( $site_url, PHP_URL_HOST );

        // SSL-Zertifikat-Ablauf per Stream-Context pruefen
        $ablauf_timestamp = null;
        $kontext = stream_context_create( array(
            'ssl' => array(
                'capture_peer_cert' => true,
                'verify_peer'       => true,
                'verify_peer_name'  => true,
            ),
        ) );

        // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fopen
        $stream = @stream_socket_client(
            'ssl://' . $host . ':443',
            $errno,
            $errstr,
            10,
            STREAM_CLIENT_CONNECT,
            $kontext
        );

        if ( $stream ) {
            $params = stream_context_get_params( $stream );
            if ( isset( $params['options']['ssl']['peer_certificate'] ) ) {
                $zertifikat_info = openssl_x509_parse( $params['options']['ssl']['peer_certificate'] );
                if ( isset( $zertifikat_info['validTo_time_t'] ) ) {
                    $ablauf_timestamp = $zertifikat_info['validTo_time_t'];
                }
            }
            // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fclose -- stream_socket_client resource, not a file
            fclose( $stream );
        }

        $tage_bis_ablauf = null;
        if ( $ablauf_timestamp ) {
            $tage_bis_ablauf = (int) round( ( $ablauf_timestamp - time() ) / DAY_IN_SECONDS );
            if ( $tage_bis_ablauf < 14 ) {
                $probleme[] = sprintf( 'SSL-Zertifikat laeuft in %d Tagen ab!', $tage_bis_ablauf );
            } elseif ( $tage_bis_ablauf < 30 ) {
                $warnungen[] = sprintf( 'SSL-Zertifikat laeuft in %d Tagen ab.', $tage_bis_ablauf );
            }
        }

        return array(
            'aktiv'           => true,
            'ablauf'          => $ablauf_timestamp ? gmdate( 'Y-m-d', $ablauf_timestamp ) : null,
            'tage_bis_ablauf' => $tage_bis_ablauf,
        );
    }

    /**
     * PHP-Version pruefen
     */
    private function php_pruefen( &$warnungen ) {
        $version     = PHP_VERSION;
        $empfohlen   = '8.1';
        $minimum     = '7.4';
        $status      = 'gut';

        if ( version_compare( $version, $minimum, '<' ) ) {
            $warnungen[] = sprintf( 'PHP %s ist veraltet (Minimum: %s).', $version, $minimum );
            $status = 'veraltet';
        } elseif ( version_compare( $version, '8.0', '<' ) ) {
            $warnungen[] = sprintf( 'PHP %s wird bald nicht mehr unterstuetzt. Empfohlen: PHP %s+.', $version, $empfohlen );
            $status = 'warnung';
        }

        return array(
            'version'   => $version,
            'empfohlen' => $empfohlen,
            'status'    => $status,
        );
    }

    /**
     * Anzahl veralteter Plugins ermitteln
     */
    private function veraltete_plugins_zaehlen() {
        $update_plugins = get_site_transient( 'update_plugins' );
        $anzahl = isset( $update_plugins->response ) ? count( $update_plugins->response ) : 0;

        return array(
            'anzahl_updates' => $anzahl,
            'status'         => $anzahl > 0 ? 'updates_verfuegbar' : 'aktuell',
        );
    }

    /**
     * Datenbank-Groesse ermitteln (in MB)
     */
    private function datenbank_groesse() {
        global $wpdb;

        $datenbank_name = DB_NAME;
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- information_schema query, caching not applicable
        $groesse = $wpdb->get_var(
            $wpdb->prepare(
                "SELECT ROUND(SUM(data_length + index_length) / 1024 / 1024, 2)
                 FROM information_schema.TABLES
                 WHERE table_schema = %s",
                $datenbank_name
            )
        );

        return array(
            'groesse_mb' => $groesse ? floatval( $groesse ) : 0,
            'datenbank'  => $datenbank_name,
        );
    }

    /**
     * Upload-Ordner-Groesse ermitteln (in MB)
     */
    private function uploads_groesse() {
        $upload_dir = wp_upload_dir();
        $pfad       = $upload_dir['basedir'];
        $groesse_bytes = 0;

        if ( is_dir( $pfad ) ) {
            $iterator = new RecursiveIteratorIterator(
                new RecursiveDirectoryIterator( $pfad, RecursiveDirectoryIterator::SKIP_DOTS )
            );
            foreach ( $iterator as $datei ) {
                if ( $datei->isFile() ) {
                    $groesse_bytes += $datei->getSize();
                }
            }
        }

        return array(
            'groesse_mb' => round( $groesse_bytes / 1024 / 1024, 2 ),
            'pfad'       => $pfad,
        );
    }

    /**
     * Kritische Dateiberechtigungen pruefen
     */
    private function dateiberechtigungen_pruefen( &$probleme ) {
        $zu_pruefende_pfade = array(
            ABSPATH . 'wp-config.php'        => array( 'max_oktal' => '0644', 'beschreibung' => 'wp-config.php' ),
            ABSPATH . '.htaccess'            => array( 'max_oktal' => '0644', 'beschreibung' => '.htaccess' ),
            WP_CONTENT_DIR . '/uploads'      => array( 'max_oktal' => '0755', 'beschreibung' => 'uploads/' ),
        );

        $ergebnisse = array();
        foreach ( $zu_pruefende_pfade as $pfad => $info ) {
            if ( ! file_exists( $pfad ) ) {
                continue;
            }
            $berechtigungen = substr( sprintf( '%o', fileperms( $pfad ) ), -4 );
            $zu_offen       = octdec( $berechtigungen ) > octdec( $info['max_oktal'] );

            if ( $zu_offen ) {
                $probleme[] = sprintf(
                    '%s hat zu offene Berechtigungen: %s (empfohlen: max. %s)',
                    $info['beschreibung'],
                    $berechtigungen,
                    $info['max_oktal']
                );
            }

            $ergebnisse[] = array(
                'datei'         => $info['beschreibung'],
                'berechtigungen' => $berechtigungen,
                'ok'            => ! $zu_offen,
            );
        }

        return $ergebnisse;
    }

    /**
     * Debug-Modus pruefen (sollte in Produktion deaktiviert sein)
     */
    private function debug_modus_pruefen( &$warnungen ) {
        $debug_aktiv = defined( 'WP_DEBUG' ) && WP_DEBUG;
        if ( $debug_aktiv ) {
            $warnungen[] = 'WP_DEBUG ist aktiv. In Produktionsumgebungen deaktivieren.';
        }
        return array( 'aktiv' => $debug_aktiv );
    }

    /**
     * WordPress-Version pruefen
     */
    private function wp_version_pruefen( &$warnungen ) {
        global $wp_version;
        $aktuelle_version = $wp_version;

        $update_core = get_site_transient( 'update_core' );
        $neueste_version = null;
        $update_verfuegbar = false;

        if ( isset( $update_core->updates ) && is_array( $update_core->updates ) ) {
            foreach ( $update_core->updates as $update ) {
                if ( 'upgrade' === $update->response ) {
                    $neueste_version   = $update->version;
                    $update_verfuegbar = true;
                    $warnungen[]       = sprintf( 'WordPress-Update verfuegbar: %s -> %s', $aktuelle_version, $neueste_version );
                    break;
                }
            }
        }

        return array(
            'version'           => $aktuelle_version,
            'neueste_version'   => $neueste_version,
            'update_verfuegbar' => $update_verfuegbar,
        );
    }

    /**
     * Server-Informationen: OS, Webserver, MySQL, PHP
     */
    private function server_info() {
        global $wpdb;

        $server_software = isset( $_SERVER['SERVER_SOFTWARE'] ) ? sanitize_text_field( wp_unslash( $_SERVER['SERVER_SOFTWARE'] ) ) : 'Unbekannt';
        // Nur den Webserver-Namen extrahieren (z.B. "Apache" aus "Apache/2.4.57")
        $webserver = explode( '/', $server_software );
        $webserver_name = $webserver[0];

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Version query, no caching needed
        $mysql_version = $wpdb->get_var( 'SELECT VERSION()' );

        return array(
            'os'              => PHP_OS_FAMILY . ' ' . php_uname( 'r' ),
            'webserver'       => $server_software,
            'webserver_name'  => $webserver_name,
            'mysql_version'   => $mysql_version ?: 'Unbekannt',
            'php_version'     => PHP_VERSION,
            'php_memory_limit' => ini_get( 'memory_limit' ),
            'php_max_input_vars' => ini_get( 'max_input_vars' ),
            'php_max_post_size'  => ini_get( 'post_max_size' ),
            'php_max_upload'     => ini_get( 'upload_max_filesize' ),
            'gd_installed'       => extension_loaded( 'gd' ),
            'zip_installed'      => extension_loaded( 'zip' ),
            'curl_installed'     => extension_loaded( 'curl' ),
            'imagick_installed'  => extension_loaded( 'imagick' ),
        );
    }

    /**
     * WordPress-Umgebungsinformationen
     */
    private function wp_environment_info() {
        global $wp_version;

        $upload_dir = wp_upload_dir();
        $max_upload = wp_max_upload_size();

        return array(
            'version'            => $wp_version,
            'site_url'           => get_site_url(),
            'home_url'           => get_home_url(),
            'multisite'          => is_multisite(),
            'max_upload_size'    => size_format( $max_upload ),
            'memory_limit'       => defined( 'WP_MEMORY_LIMIT' ) ? WP_MEMORY_LIMIT : '40M',
            'max_memory_limit'   => defined( 'WP_MAX_MEMORY_LIMIT' ) ? WP_MAX_MEMORY_LIMIT : '256M',
            'permalink_structure' => get_option( 'permalink_structure' ) ?: 'Standard',
            'language'           => get_locale(),
            'timezone'           => wp_timezone_string(),
            'admin_email'        => get_option( 'admin_email' ),
            'debug_mode'         => defined( 'WP_DEBUG' ) && WP_DEBUG,
            'cron_enabled'       => ! ( defined( 'DISABLE_WP_CRON' ) && DISABLE_WP_CRON ),
            'uploads_writable'   => wp_is_writable( $upload_dir['basedir'] ),
        );
    }

    /**
     * Theme-Informationen
     */
    private function theme_info() {
        $theme = wp_get_theme();
        $parent = $theme->parent();

        $info = array(
            'name'           => $theme->get( 'Name' ),
            'version'        => $theme->get( 'Version' ),
            'author'         => $theme->get( 'Author' ),
            'is_child_theme' => is_child_theme(),
        );

        if ( $parent ) {
            $info['parent_name']    = $parent->get( 'Name' );
            $info['parent_version'] = $parent->get( 'Version' );
            $info['parent_author']  = $parent->get( 'Author' );
        }

        return $info;
    }
}
