<?php
/**
 * Code-Snippets Verwaltung
 * Snippets werden in wp_options gespeichert und in ein Custom Plugin geschrieben
 *
 * @package WP_Grid_Connector
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class WPGrid_Snippets {

    const OPTION_KEY    = 'wpgrid_snippets';
    const PLUGIN_DATEI  = 'wpgrid-custom-snippets/wpgrid-custom-snippets.php';

    /**
     * GET /snippets - Alle gespeicherten Snippets auflisten
     */
    public function snippets_listen( WP_REST_Request $request ) {
        $snippets = $this->snippets_laden();
        return wpgrid_antwort( true, array(
            'snippets' => array_values( $snippets ),
            'anzahl'   => count( $snippets ),
        ) );
    }

    /**
     * POST /snippets/add - Neues Snippet hinzufuegen
     */
    public function snippet_hinzufuegen( WP_REST_Request $request ) {
        $titel = $request->get_param( 'title' );
        $code  = $request->get_param( 'code' );

        if ( empty( $titel ) || empty( $code ) ) {
            return wpgrid_antwort( false, null, 'Titel und Code sind erforderlich.' );
        }

        // PHP-Syntax pruefen
        $syntax_fehler = $this->php_syntax_pruefen( $code );
        if ( $syntax_fehler ) {
            return wpgrid_antwort( false, null, 'PHP-Syntaxfehler: ' . $syntax_fehler );
        }

        $snippets = $this->snippets_laden();
        $neue_id  = time() . '_' . substr( md5( $titel ), 0, 6 );

        $snippets[ $neue_id ] = array(
            'id'          => $neue_id,
            'titel'       => sanitize_text_field( $titel ),
            'code'        => $code, // Code wird nicht escaped - wird in PHP-Datei geschrieben
            'erstellt_am' => current_time( 'c' ),
            'aktiv'       => true,
        );

        update_option( self::OPTION_KEY, $snippets );
        $this->custom_plugin_schreiben( $snippets );

        wpgrid_debug( 'Snippet hinzugefuegt', array( 'id' => $neue_id, 'titel' => $titel ) );
        return wpgrid_antwort( true, array( 'id' => $neue_id, 'titel' => $titel ) );
    }

    /**
     * DELETE /snippets/remove - Snippet entfernen
     */
    public function snippet_entfernen( WP_REST_Request $request ) {
        $id       = (string) $request->get_param( 'id' );
        $snippets = $this->snippets_laden();

        if ( ! isset( $snippets[ $id ] ) ) {
            return wpgrid_antwort( false, null, 'Snippet nicht gefunden: ' . esc_html( $id ) );
        }

        $titel = $snippets[ $id ]['titel'];
        unset( $snippets[ $id ] );

        update_option( self::OPTION_KEY, $snippets );
        $this->custom_plugin_schreiben( $snippets );

        wpgrid_debug( 'Snippet entfernt', array( 'id' => $id ) );
        return wpgrid_antwort( true, array( 'entfernt' => $id, 'titel' => $titel ) );
    }

    // â”€â”€ Hilfsmethoden â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€

    /**
     * Snippets aus wp_options laden
     */
    private function snippets_laden() {
        $snippets = get_option( self::OPTION_KEY, array() );
        return is_array( $snippets ) ? $snippets : array();
    }

    /**
     * Custom-Plugin-Datei mit allen aktiven Snippets neu schreiben
     * Sicherer als functions.php zu modifizieren
     */
    private function custom_plugin_schreiben( $snippets ) {
        if ( ! function_exists( 'WP_Filesystem' ) ) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
        }

        global $wp_filesystem;
        WP_Filesystem();

        $plugin_verzeichnis = WP_PLUGIN_DIR . '/wpgrid-custom-snippets';
        $plugin_datei_pfad  = $plugin_verzeichnis . '/wpgrid-custom-snippets.php';

        // Verzeichnis erstellen falls nicht vorhanden
        if ( ! $wp_filesystem->is_dir( $plugin_verzeichnis ) ) {
            $wp_filesystem->mkdir( $plugin_verzeichnis, 0755 );
        }

        // Plugin-Header + aktive Snippets zusammenstellen
        $inhalt  = "<?php\n";
        $inhalt .= "/**\n";
        $inhalt .= " * Plugin Name: WP-Grid Custom Snippets\n";
        $inhalt .= " * Description: Automatisch generiert von Grid Connector. Nicht manuell bearbeiten.\n";
        $inhalt .= " * Version: 1.0\n";
        $inhalt .= " * Author: WP-Grid\n";
        $inhalt .= " */\n\n";
        $inhalt .= "if ( ! defined( 'ABSPATH' ) ) { exit; }\n\n";

        $aktive_snippets = array_filter( $snippets, function ( $s ) {
            return ! empty( $s['aktiv'] );
        } );

        foreach ( $aktive_snippets as $snippet ) {
            $inhalt .= "// Snippet: " . sanitize_text_field( $snippet['titel'] ) . " (ID: " . $snippet['id'] . ")\n";
            $inhalt .= $snippet['code'] . "\n\n";
        }

        $wp_filesystem->put_contents( $plugin_datei_pfad, $inhalt, FS_CHMOD_FILE );

        // Plugin aktivieren falls noch nicht aktiv
        if ( ! is_plugin_active( self::PLUGIN_DATEI ) && ! empty( $aktive_snippets ) ) {
            activate_plugin( self::PLUGIN_DATEI );
        }

        // Plugin deaktivieren wenn keine Snippets mehr vorhanden
        if ( empty( $aktive_snippets ) && is_plugin_active( self::PLUGIN_DATEI ) ) {
            deactivate_plugins( self::PLUGIN_DATEI );
        }
    }

    /**
     * PHP-Syntax mit php -l pruefen (falls verfuegbar)
     */
    private function php_syntax_pruefen( $code ) {
        // Temporaere Datei erstellen
        $temp_datei = wp_tempnam( 'wpgrid_snippet_' );
        // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_file_put_contents
        file_put_contents( $temp_datei, "<?php\n" . $code );

        $ausgabe     = array();
        $rueckgabe   = 0;
        // phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.system_calls_exec
        exec( 'php -l ' . escapeshellarg( $temp_datei ) . ' 2>&1', $ausgabe, $rueckgabe );

        // phpcs:ignore WordPress.WP.AlternativeFunctions.unlink_unlink
        @unlink( $temp_datei );

        if ( 0 !== $rueckgabe ) {
            return implode( ' ', $ausgabe );
        }

        return null;
    }
}
