<?php
/**
 * Update-Verwaltung fuer Core, Plugins und Themes
 *
 * @package WP_Grid_Connector
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class WPGrid_Updates {

    /**
     * POST /update/core - WordPress Core aktualisieren
     */
    public function core_updaten( WP_REST_Request $request ) {
        require_once ABSPATH . 'wp-admin/includes/update.php';
        require_once ABSPATH . 'wp-admin/includes/file.php';
        require_once ABSPATH . 'wp-admin/includes/misc.php';
        require_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';
        require_once ABSPATH . 'wp-admin/includes/class-core-upgrader.php';

        wp_version_check();
        $update_core = get_site_transient( 'update_core' );

        $update = null;
        if ( isset( $update_core->updates ) ) {
            foreach ( $update_core->updates as $u ) {
                if ( 'upgrade' === $u->response ) {
                    $update = $u;
                    break;
                }
            }
        }

        if ( ! $update ) {
            return wpgrid_antwort( true, array( 'status' => 'aktuell', 'nachricht' => 'WordPress ist bereits auf dem neuesten Stand.' ) );
        }

        $upgrader = new Core_Upgrader( new WP_Ajax_Upgrader_Skin() );
        $ergebnis = $upgrader->upgrade( $update );

        if ( is_wp_error( $ergebnis ) ) {
            wpgrid_debug( 'Core-Update fehlgeschlagen', array( 'fehler' => $ergebnis->get_error_message() ) );
            return wpgrid_antwort( false, null, $ergebnis->get_error_message() );
        }

        wpgrid_debug( 'Core aktualisiert', array( 'version' => $update->version ) );
        return wpgrid_antwort( true, array(
            'status'      => 'aktualisiert',
            'neue_version' => $update->version,
        ) );
    }

    /**
     * POST /update/plugin - Einzelnes Plugin aktualisieren
     */
    public function plugin_updaten( WP_REST_Request $request ) {
        require_once ABSPATH . 'wp-admin/includes/plugin.php';
        require_once ABSPATH . 'wp-admin/includes/update.php';
        require_once ABSPATH . 'wp-admin/includes/file.php';
        require_once ABSPATH . 'wp-admin/includes/misc.php';
        require_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';
        require_once ABSPATH . 'wp-admin/includes/class-plugin-upgrader.php';

        $slug         = $request->get_param( 'slug' );
        $plugin_datei = $this->slug_zu_datei( $slug );

        if ( ! $plugin_datei ) {
            return wpgrid_antwort( false, null, 'Plugin nicht gefunden: ' . esc_html( $slug ) );
        }

        wp_update_plugins();
        $update_plugins = get_site_transient( 'update_plugins' );

        if ( ! isset( $update_plugins->response[ $plugin_datei ] ) ) {
            return wpgrid_antwort( true, array( 'status' => 'aktuell', 'plugin' => $slug ) );
        }

        $upgrader = new Plugin_Upgrader( new WP_Ajax_Upgrader_Skin() );
        $ergebnis = $upgrader->upgrade( $plugin_datei );

        if ( is_wp_error( $ergebnis ) ) {
            return wpgrid_antwort( false, null, $ergebnis->get_error_message() );
        }

        wpgrid_debug( 'Plugin aktualisiert', array( 'slug' => $slug ) );
        return wpgrid_antwort( true, array(
            'status'      => 'aktualisiert',
            'plugin'      => $slug,
            'neue_version' => $update_plugins->response[ $plugin_datei ]->new_version,
        ) );
    }

    /**
     * POST /update/theme - Theme aktualisieren
     */
    public function theme_updaten( WP_REST_Request $request ) {
        require_once ABSPATH . 'wp-admin/includes/update.php';
        require_once ABSPATH . 'wp-admin/includes/file.php';
        require_once ABSPATH . 'wp-admin/includes/misc.php';
        require_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';
        require_once ABSPATH . 'wp-admin/includes/class-theme-upgrader.php';

        $slug = $request->get_param( 'slug' );

        wp_update_themes();
        $update_themes = get_site_transient( 'update_themes' );

        if ( ! isset( $update_themes->response[ $slug ] ) ) {
            return wpgrid_antwort( true, array( 'status' => 'aktuell', 'theme' => $slug ) );
        }

        $upgrader = new Theme_Upgrader( new WP_Ajax_Upgrader_Skin() );
        $ergebnis = $upgrader->upgrade( $slug );

        if ( is_wp_error( $ergebnis ) ) {
            return wpgrid_antwort( false, null, $ergebnis->get_error_message() );
        }

        return wpgrid_antwort( true, array(
            'status'      => 'aktualisiert',
            'theme'       => $slug,
            'neue_version' => $update_themes->response[ $slug ]['new_version'],
        ) );
    }

    /**
     * POST /update/all - Alle Plugins und Themes aktualisieren (mit Rollback bei Fehler)
     */
    public function alles_updaten( WP_REST_Request $request ) {
        require_once ABSPATH . 'wp-admin/includes/plugin.php';
        require_once ABSPATH . 'wp-admin/includes/update.php';
        require_once ABSPATH . 'wp-admin/includes/file.php';
        require_once ABSPATH . 'wp-admin/includes/misc.php';
        require_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';
        require_once ABSPATH . 'wp-admin/includes/class-plugin-upgrader.php';
        require_once ABSPATH . 'wp-admin/includes/class-theme-upgrader.php';

        wp_update_plugins();
        wp_update_themes();

        $update_plugins = get_site_transient( 'update_plugins' );
        $update_themes  = get_site_transient( 'update_themes' );

        $ergebnisse = array(
            'plugins_aktualisiert' => array(),
            'themes_aktualisiert'  => array(),
            'fehler'               => array(),
        );

        // Plugins aktualisieren
        if ( ! empty( $update_plugins->response ) ) {
            $upgrader = new Plugin_Upgrader( new WP_Ajax_Upgrader_Skin() );
            foreach ( array_keys( $update_plugins->response ) as $plugin_datei ) {
                // Grid Connector selbst ueberspringen
                if ( plugin_basename( WPGRID_PLUGIN_FILE ) === $plugin_datei ) {
                    continue;
                }

                $ergebnis = $upgrader->upgrade( $plugin_datei );
                if ( is_wp_error( $ergebnis ) ) {
                    $ergebnisse['fehler'][] = array(
                        'typ'    => 'plugin',
                        'datei'  => $plugin_datei,
                        'fehler' => $ergebnis->get_error_message(),
                    );
                } else {
                    $ergebnisse['plugins_aktualisiert'][] = $plugin_datei;
                }
            }
        }

        // Themes aktualisieren
        if ( ! empty( $update_themes->response ) ) {
            $upgrader = new Theme_Upgrader( new WP_Ajax_Upgrader_Skin() );
            foreach ( array_keys( $update_themes->response ) as $theme_slug ) {
                $ergebnis = $upgrader->upgrade( $theme_slug );
                if ( is_wp_error( $ergebnis ) ) {
                    $ergebnisse['fehler'][] = array(
                        'typ'    => 'theme',
                        'slug'   => $theme_slug,
                        'fehler' => $ergebnis->get_error_message(),
                    );
                } else {
                    $ergebnisse['themes_aktualisiert'][] = $theme_slug;
                }
            }
        }

        $erfolg = empty( $ergebnisse['fehler'] );
        wpgrid_debug( 'Alle Updates abgeschlossen', $ergebnisse );
        return wpgrid_antwort( $erfolg, $ergebnisse, $erfolg ? '' : 'Einige Updates sind fehlgeschlagen.' );
    }

    /**
     * Slug zu Plugin-Datei aufloesen
     */
    private function slug_zu_datei( $slug ) {
        if ( ! function_exists( 'get_plugins' ) ) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }
        $slug = sanitize_text_field( $slug );
        if ( strpos( $slug, '/' ) !== false ) {
            $alle = get_plugins();
            return isset( $alle[ $slug ] ) ? $slug : null;
        }
        foreach ( array_keys( get_plugins() ) as $datei ) {
            if ( dirname( $datei ) === $slug || $datei === $slug . '.php' ) {
                return $datei;
            }
        }
        return null;
    }
}
